﻿/*******************************************************************************
 * token.js, Copyright (c) Daniel Schoeberle 2014                               *
 *                                                                             *
 * User library for accessing PKCS #11 API .dll library from JavaScript        *
 * Requires js-ctypes and chrome authority, now only available from FireFox    *
 * add-n context.                                                              *
 *******************************************************************************/

const {Cc,Cu,Ci} = require("chrome");
Cu.import("resource://gre/modules/ctypes.jsm");

var prompts = Cc["@mozilla.org/embedcomp/prompt-service;1"].getService(Ci.nsIPromptService);
/*********** Start Definitions ***********/
/* Constants */
const MAX_OBJECT_SEARCH = 100;	// loop maximum number of times when searching for objects with C_FindObjects

// session
const CKF_RW_SESSION = 0x00000002;
const CKF_SERIAL_SESSION = 0x00000004;

// user type
const CKU_SO = 0;
const CKU_USER = 1;
const CKU_CONTECT_SPECIFIC = 2;

// function return value
const CKR_OK 								= 0x00000000;
const CKR_CANCEL 							= 0x00000001;
const CKR_HOST_MEMORY 						= 0x00000002;
const CKR_SLOT_ID_INVALID 					= 0x00000003;
const CKR_GENERAL_ERROR 					= 0x00000005;
const CKR_FUNCTION_FAILED 					= 0x00000006;
const CKR_ARGUMENTS_BAD 					= 0x00000007;
const CKR_NO_EVENT 							= 0x00000008;
const CKR_NEED_TO_CREATE_THREADS 			= 0x00000009;
const CKR_CANT_LOCK 						= 0x0000000A;
const CKR_ATTRIBUTE_READ_ONLY 				= 0x00000010;
const CKR_ATTRIBUTE_SENSITIVE 				= 0x00000011;
const CKR_ATTRIBUTE_TYPE_INVALID 			= 0x00000012;
const CKR_ATTRIBUTE_VALUE_INVALID 			= 0x00000013;
const CKR_DATA_INVALID 						= 0x00000020;
const CKR_DATA_LEN_RANGE 					= 0x00000021;
const CKR_DEVICE_ERROR 						= 0x00000030;
const CKR_DEVICE_MEMORY 					= 0x00000031;
const CKR_DEVICE_REMOVED 					= 0x00000032;
const CKR_ENCRYPTED_DATA_INVALID 			= 0x00000040;
const CKR_ENCRYPTED_DATA_LEN_RANGE 			= 0x00000041;
const CKR_FUNCTION_CANCELED 				= 0x00000050;
const CKR_FUNCTION_NOT_PARALLEL 			= 0x00000051;
const CKR_FUNCTION_NOT_SUPPORTED 			= 0x00000054;
const CKR_KEY_HANDLE_INVALID 				= 0x00000060;
const CKR_KEY_SIZE_RANGE 					= 0x00000062;
const CKR_KEY_TYPE_INCONSISTENT 			= 0x00000063;
const CKR_KEY_NOT_NEEDED 					= 0x00000064;
const CKR_KEY_CHANGED 						= 0x00000065;
const CKR_KEY_NEEDED 						= 0x00000066;
const CKR_KEY_INDIGESTIBLE 					= 0x00000067;
const CKR_KEY_FUNCTION_NOT_PERMITTED 		= 0x00000068;
const CKR_KEY_NOT_WRAPPABLE 				= 0x00000069;
const CKR_KEY_UNEXTRACTABLE 				= 0x0000006A;
const CKR_MECHANISM_INVALID 				= 0x00000070;
const CKR_MECHANISM_PARAM_INVALID 			= 0x00000071;
const CKR_OBJECT_HANDLE_INVALID 			= 0x00000082;
const CKR_OPERATION_ACTIVE 					= 0x00000090;
const CKR_OPERATION_NOT_INITIALIZED 		= 0x00000091;
const CKR_PIN_INCORRECT 					= 0x000000A0;
const CKR_PIN_INVALID 						= 0x000000A1;
const CKR_PIN_LEN_RANGE 					= 0x000000A2;
const CKR_PIN_EXPIRED 						= 0x000000A3;
const CKR_PIN_LOCKED 						= 0x000000A4;
const CKR_SESSION_CLOSED 					= 0x000000B0;
const CKR_SESSION_COUNT 					= 0x000000B1;
const CKR_SESSION_HANDLE_INVALID 			= 0x000000B3;
const CKR_SESSION_PARALLEL_NOT_SUPPORTED 	= 0x000000B4;
const CKR_SESSION_READ_ONLY 				= 0x000000B5;
const CKR_SESSION_EXISTS 					= 0x000000B6;
const CKR_SESSION_READ_ONLY_EXISTS 			= 0x000000B7;
const CKR_SESSION_READ_WRITE_SO_EXISTS 		= 0x000000B8;
const CKR_SIGNATURE_INVALID 				= 0x000000C0;
const CKR_SIGNATURE_LEN_RANGE 				= 0x000000C1;
const CKR_TEMPLATE_INCOMPLETE 				= 0x000000D0;
const CKR_TEMPLATE_INCONSISTENT 			= 0x000000D1;
const CKR_TOKEN_NOT_PRESENT 				= 0x000000E0;
const CKR_TOKEN_NOT_RECOGNIZED 				= 0x000000E1;
const CKR_TOKEN_WRITE_PROTECTED 			= 0x000000E2;
const CKR_UNWRAPPING_KEY_HANDLE_INVALID 	= 0x000000F0;
const CKR_UNWRAPPING_KEY_SIZE_RANGE 		= 0x000000F1;
const CKR_UNWRAPPING_KEY_TYPE_INCONSISTENT 	= 0x000000F2;
const CKR_USER_ALREADY_LOGGED_IN 			= 0x00000100;
const CKR_USER_NOT_LOGGED_IN 				= 0x00000101;
const CKR_USER_PIN_NOT_INITIALIZED 			= 0x00000102;
const CKR_USER_TYPE_INVALID 				= 0x00000103;
const CKR_USER_ANOTHER_ALREADY_LOGGED_IN 	= 0x00000104;
const CKR_USER_TOO_MANY_TYPES 				= 0x00000105;
const CKR_WRAPPED_KEY_INVALID 				= 0x00000110;
const CKR_WRAPPED_KEY_LEN_RANGE 			= 0x00000112;
const CKR_WRAPPING_KEY_HANDLE_INVALID 		= 0x00000113;
const CKR_WRAPPING_KEY_SIZE_RANGE 			= 0x00000114;
const CKR_WRAPPING_KEY_TYPE_INCONSISTENT 	= 0x00000115;
const CKR_RANDOM_SEED_NOT_SUPPORTED 		= 0x00000120;
const CKR_RANDOM_NO_RNG 					= 0x00000121;
const CKR_DOMAIN_PARAMS_INVALID 			= 0x00000130;
const CKR_BUFFER_TOO_SMALL 					= 0x00000150;
const CKR_SAVED_STATE_INVALID 				= 0x00000160;
const CKR_INFORMATION_SENSITIVE 			= 0x00000170;
const CKR_STATE_UNSAVEABLE 					= 0x00000180;
const CKR_CRYPTOKI_NOT_INITIALIZED 			= 0x00000190;
const CKR_CRYPTOKI_ALREADY_INITIALIZED 		= 0x00000191;
const CKR_MUTEX_BAD 						= 0x000001A0;
const CKR_MUTEX_NOT_LOCKED 					= 0x000001A1;
const CKR_FUNCTION_REJECTED 				= 0x00000200;
const CKR_VENDOR_DEFINED 					= 0x80000000;

// object class
const CKO_DATA              = 0x00000000;
const CKO_CERTIFICATE       = 0x00000001;
const CKO_PUBLIC_KEY        = 0x00000002;
const CKO_PRIVATE_KEY       = 0x00000003;
const CKO_SECRET_KEY        = 0x00000004;
const CKO_HW_FEATURE        = 0x00000005;
const CKO_DOMAIN_PARAMETERS	= 0x00000006;
const CKO_MECHANISM         = 0x00000007;
const CKO_VENDOR_DEFINED    = 0x80000000; 

// certificate types
const CKC_X_509				= 0x00000000;
const CKC_X_509_ATTR_CERT	= 0x00000001;
const CKC_WTLS				= 0x00000002;
const CKC_VENDOR_DEFINED	= 0x80000000;

// mechanisms
const CKM_RSA_PKCS_KEY_PAIR_GEN 			= 0x00000000;
const CKM_RSA_PKCS 							= 0x00000001;
const CKM_RSA_9796 							= 0x00000002;
const CKM_RSA_X_509 						= 0x00000003;
const CKM_MD2_RSA_PKCS 						= 0x00000004;
const CKM_MD5_RSA_PKCS 						= 0x00000005;
const CKM_SHA1_RSA_PKCS 					= 0x00000006;
const CKM_RIPEMD128_RSA_PKCS				= 0x00000007;
const CKM_RIPEMD160_RSA_PKCS 				= 0x00000008;
const CKM_RSA_PKCS_OAEP 					= 0x00000009;
const CKM_RSA_X9_31_KEY_PAIR_GEN 			= 0x0000000A;
const CKM_RSA_X9_31 						= 0x0000000B;
const CKM_SHA1_RSA_X9_31 					= 0x0000000C;
const CKM_RSA_PKCS_PSS 						= 0x0000000D;
const CKM_SHA1_RSA_PKCS_PSS 				= 0x0000000E;
const CKM_DSA_KEY_PAIR_GEN 					= 0x00000010;
const CKM_DSA 								= 0x00000011;
const CKM_DSA_SHA1 							= 0x00000012;
const CKM_DH_PKCS_KEY_PAIR_GEN 				= 0x00000020;
const CKM_DH_PKCS_DERIVE 					= 0x00000021;
const CKM_X9_42_DH_KEY_PAIR_GEN 			= 0x00000030;
const CKM_X9_42_DH_DERIVE 					= 0x00000031;
const CKM_X9_42_DH_HYBRID_DERIVE 			= 0x00000032;
const CKM_X9_42_MQV_DERIVE 					= 0x00000033;
const CKM_SHA256_RSA_PKCS 					= 0x00000040;
const CKM_SHA384_RSA_PKCS 					= 0x00000041;
const CKM_SHA512_RSA_PKCS 					= 0x00000042;
const CKM_SHA256_RSA_PKCS_PSS 				= 0x00000043;
const CKM_SHA384_RSA_PKCS_PSS 				= 0x00000044;
const CKM_SHA512_RSA_PKCS_PSS 				= 0x00000045;
const CKM_RC2_KEY_GEN 						= 0x00000100;
const CKM_RC2_ECB 							= 0x00000101;
const CKM_RC2_CBC 							= 0x00000102;
const CKM_RC2_MAC 							= 0x00000103;
const CKM_RC2_MAC_GENERAL 					= 0x00000104;
const CKM_RC2_CBC_PAD 						= 0x00000105;
const CKM_RC4_KEY_GEN 						= 0x00000110;
const CKM_RC4 								= 0x00000111;
const CKM_DES_KEY_GEN 						= 0x00000120;
const CKM_DES_ECB 							= 0x00000121;
const CKM_DES_CBC 							= 0x00000122;
const CKM_DES_MAC 							= 0x00000123;
const CKM_DES_MAC_GENERAL 					= 0x00000124;
const CKM_DES_CBC_PAD						= 0x00000125;
const CKM_DES2_KEY_GEN 						= 0x00000130;
const CKM_DES3_KEY_GEN 						= 0x00000131;
const CKM_DES3_ECB 							= 0x00000132;
const CKM_DES3_CBC 							= 0x00000133;
const CKM_DES3_MAC 							= 0x00000134;
const CKM_DES3_MAC_GENERAL 					= 0x00000135;
const CKM_DES3_CBC_PAD 						= 0x00000136;
const CKM_CDMF_KEY_GEN 						= 0x00000140;
const CKM_CDMF_ECB 							= 0x00000141;
const CKM_CDMF_CBC 							= 0x00000142;
const CKM_CDMF_MAC 							= 0x00000143;
const CKM_CDMF_MAC_GENERAL 					= 0x00000144;
const CKM_CDMF_CBC_PAD 						= 0x00000145;
const CKM_DES_OFB64 						= 0x00000150;
const CKM_DES_OFB8 							= 0x00000151;
const CKM_DES_CFB64 						= 0x00000152;
const CKM_DES_CFB8 							= 0x00000153;
const CKM_MD2 								= 0x00000200;
const CKM_MD2_HMAC 							= 0x00000201;
const CKM_MD2_HMAC_GENERAL 					= 0x00000202;
const CKM_MD5 								= 0x00000210;
const CKM_MD5_HMAC 							= 0x00000211;
const CKM_MD5_HMAC_GENERAL 					= 0x00000212;
const CKM_SHA_1 							= 0x00000220;
const CKM_SHA_1_HMAC 						= 0x00000221;
const CKM_SHA_1_HMAC_GENERAL 				= 0x00000222;
const CKM_RIPEMD128 						= 0x00000230;
const CKM_RIPEMD128_HMAC 					= 0x00000231;
const CKM_RIPEMD128_HMAC_GENERAL 			= 0x00000232;
const CKM_RIPEMD160 						= 0x00000240;
const CKM_RIPEMD160_HMAC 					= 0x00000241;
const CKM_RIPEMD160_HMAC_GENERAL			= 0x00000242;
const CKM_SHA256 							= 0x00000250;
const CKM_SHA256_HMAC 						= 0x00000251;
const CKM_SHA256_HMAC_GENERAL				= 0x00000252;
const CKM_SHA384 							= 0x00000260;
const CKM_SHA384_HMAC 						= 0x00000261;
const CKM_SHA384_HMAC_GENERAL 				= 0x00000262;
const CKM_SHA512 							= 0x00000270;
const CKM_SHA512_HMAC 						= 0x00000271;
const CKM_SHA512_HMAC_GENERAL 				= 0x00000272;
const CKM_CAST_KEY_GEN 						= 0x00000300;
const CKM_CAST_ECB 							= 0x00000301;
const CKM_CAST_CBC 							= 0x00000302;
const CKM_CAST_MAC 							= 0x00000303;
const CKM_CAST_MAC_GENERAL 					= 0x00000304;
const CKM_CAST_CBC_PAD 						= 0x00000305;
const CKM_CAST3_KEY_GEN 					= 0x00000310;
const CKM_CAST3_ECB 						= 0x00000311;
const CKM_CAST3_CBC 						= 0x00000312;
const CKM_CAST3_MAC 						= 0x00000313;
const CKM_CAST3_MAC_GENERAL 				= 0x00000314;
const CKM_CAST3_CBC_PAD 					= 0x00000315;
const CKM_CAST5_KEY_GEN 					= 0x00000320;
const CKM_CAST128_KEY_GEN 					= 0x00000320;
const CKM_CAST5_ECB 						= 0x00000321;
const CKM_CAST128_ECB 						= 0x00000321;
const CKM_CAST5_CBC 						= 0x00000322;
const CKM_CAST128_CBC 						= 0x00000322;
const CKM_CAST5_MAC 						= 0x00000323;
const CKM_CAST128_MAC 						= 0x00000323;
const CKM_CAST5_MAC_GENERAL 				= 0x00000324;
const CKM_CAST128_MAC_GENERAL 				= 0x00000324;
const CKM_CAST5_CBC_PAD 					= 0x00000325;
const CKM_CAST128_CBC_PAD 					= 0x00000325;
const CKM_RC5_KEY_GEN 						= 0x00000330;
const CKM_RC5_ECB 							= 0x00000331;
const CKM_RC5_CBC 							= 0x00000332;
const CKM_RC5_MAC 							= 0x00000333;
const CKM_RC5_MAC_GENERAL 					= 0x00000334;
const CKM_RC5_CBC_PAD 						= 0x00000335;
const CKM_IDEA_KEY_GEN 						= 0x00000340;
const CKM_IDEA_ECB 							= 0x00000341;
const CKM_IDEA_CBC 							= 0x00000342;
const CKM_IDEA_MAC 							= 0x00000343;
const CKM_IDEA_MAC_GENERAL 					= 0x00000344;
const CKM_IDEA_CBC_PAD 						= 0x00000345;
const CKM_GENERIC_SECRET_KEY_GEN 			= 0x00000350;
const CKM_CONCATENATE_BASE_AND_KEY 			= 0x00000360;
const CKM_CONCATENATE_BASE_AND_DATA 		= 0x00000362;
const CKM_CONCATENATE_DATA_AND_BASE 		= 0x00000363;
const CKM_XOR_BASE_AND_DATA 				= 0x00000364;
const CKM_EXTRACT_KEY_FROM_KEY 				= 0x00000365;
const CKM_SSL3_PRE_MASTER_KEY_GEN 			= 0x00000370;
const CKM_SSL3_MASTER_KEY_DERIVE 			= 0x00000371;
const CKM_SSL3_KEY_AND_MAC_DERIVE 			= 0x00000372;
const CKM_SSL3_MASTER_KEY_DERIVE_DH 		= 0x00000373;
const CKM_TLS_PRE_MASTER_KEY_GEN 			= 0x00000374;
const CKM_TLS_MASTER_KEY_DERIVE 			= 0x00000375;
const CKM_TLS_KEY_AND_MAC_DERIVE 			= 0x00000376;
const CKM_TLS_MASTER_KEY_DERIVE_DH 			= 0x00000377;
const CKM_TLS_PRF 							= 0x00000378;
const CKM_SSL3_MD5_MAC						= 0x00000380;
const CKM_SSL3_SHA1_MAC 					= 0x00000381;
const CKM_MD5_KEY_DERIVATION 				= 0x00000390;
const CKM_MD2_KEY_DERIVATION 				= 0x00000391;
const CKM_SHA1_KEY_DERIVATION 				= 0x00000392;
const CKM_SHA256_KEY_DERIVATION 			= 0x00000393;
const CKM_SHA384_KEY_DERIVATION 			= 0x00000394;
const CKM_SHA512_KEY_DERIVATION 			= 0x00000395;
const CKM_PBE_MD2_DES_CBC 					= 0x000003A0;
const CKM_PBE_MD5_DES_CBC 					= 0x000003A1;
const CKM_PBE_MD5_CAST_CBC 					= 0x000003A2;
const CKM_PBE_MD5_CAST3_CBC 				= 0x000003A3;
const CKM_PBE_MD5_CAST5_CBC 				= 0x000003A4;
const CKM_PBE_MD5_CAST128_CBC 				= 0x000003A4;
const CKM_PBE_SHA1_CAST5_CBC 				= 0x000003A5;
const CKM_PBE_SHA1_CAST128_CBC 				= 0x000003A5;
const CKM_PBE_SHA1_RC4_128 					= 0x000003A6;
const CKM_PBE_SHA1_RC4_40 					= 0x000003A7;
const CKM_PBE_SHA1_DES3_EDE_CBC 			= 0x000003A8;
const CKM_PBE_SHA1_DES2_EDE_CBC 			= 0x000003A9;
const CKM_PBE_SHA1_RC2_128_CBC 				= 0x000003AA;
const CKM_PBE_SHA1_RC2_40_CBC 				= 0x000003AB;
const CKM_PKCS5_PBKD2 						= 0x000003B0;
const CKM_PBA_SHA1_WITH_SHA1_HMAC 			= 0x000003C0;
const CKM_WTLS_PRE_MASTER_KEY_GEN 			= 0x000003D0;
const CKM_WTLS_MASTER_KEY_DERIVE 			= 0x000003D1;
const CKM_WTLS_MASTER_KEY_DERVIE_DH_ECC		= 0x000003D2;
const CKM_WTLS_PRF 							= 0x000003D3;
const CKM_WTLS_SERVER_KEY_AND_MAC_DERIVE	= 0x000003D4;
const CKM_WTLS_CLIENT_KEY_AND_MAC_DERIVE	= 0x000003D5;
const CKM_KEY_WRAP_LYNKS 					= 0x00000400;
const CKM_KEY_WRAP_SET_OAEP 				= 0x00000401;
const CKM_CMS_SIG 							= 0x00000500;
const CKM_SKIPJACK_KEY_GEN 					= 0x00001000;
const CKM_SKIPJACK_ECB64 					= 0x00001001;
const CKM_SKIPJACK_CBC64 					= 0x00001002;
const CKM_SKIPJACK_OFB64 					= 0x00001003;
const CKM_SKIPJACK_CFB64 					= 0x00001004;
const CKM_SKIPJACK_CFB32 					= 0x00001005;
const CKM_SKIPJACK_CFB16 					= 0x00001006;
const CKM_SKIPJACK_CFB8 					= 0x00001007;
const CKM_SKIPJACK_WRAP 					= 0x00001008;
const CKM_SKIPJACK_PRIVATE_WRAP 			= 0x00001009;
const CKM_SKIPJACK_RELAYX 					= 0x0000100a;
const CKM_KEA_KEY_PAIR_GEN 					= 0x00001010;
const CKM_KEA_KEY_DERIVE 					= 0x00001011;
const CKM_FORTEZZA_TIMESTAMP 				= 0x00001020;
const CKM_BATON_KEY_GEN 					= 0x00001030;
const CKM_BATON_ECB128 						= 0x00001031;
const CKM_BATON_ECB96 						= 0x00001032;
const CKM_BATON_CBC128 						= 0x00001033;
const CKM_BATON_COUNTER 					= 0x00001034;
const CKM_BATON_SHUFFLE 					= 0x00001035;
const CKM_BATON_WRAP 						= 0x00001036;
const CKM_ECDSA_KEY_PAIR_GEN 				= 0x00001040;
const CKM_EC_KEY_PAIR_GEN 					= 0x00001040;
const CKM_ECDSA 							= 0x00001041;
const CKM_ECDSA_SHA1 						= 0x00001042;
const CKM_ECDH1_DERIVE 						= 0x00001050;
const CKM_ECDH1_COFACTOR_DERIVE 			= 0x00001051;
const CKM_ECMQV_DERIVE 						= 0x00001052;
const CKM_JUNIPER_KEY_GEN 					= 0x00001060;
const CKM_JUNIPER_ECB128 					= 0x00001061;
const CKM_JUNIPER_CBC128 					= 0x00001062;
const CKM_JUNIPER_COUNTER 					= 0x00001063;
const CKM_JUNIPER_SHUFFLE 					= 0x00001064;
const CKM_JUNIPER_WRAP 						= 0x00001065;
const CKM_FASTHASH 							= 0x00001070;
const CKM_AES_KEY_GEN 						= 0x00001080;
const CKM_AES_ECB 							= 0x00001081;
const CKM_AES_CBC 							= 0x00001082;
const CKM_AES_MAC 							= 0x00001083;
const CKM_AES_MAC_GENERAL 					= 0x00001084;
const CKM_AES_CBC_PAD 						= 0x00001085;
const CKM_BLOWFISH_KEY_GEN 					= 0x00001090;
const CKM_BLOWFISH_CBC 						= 0x00001091;
const CKM_TWOFISH_KEY_GEN 					= 0x00001092;
const CKM_TWOFISH_CBC 						= 0x00001093;
const CKM_DES_ECB_ENCRYPT_DATA 				= 0x00001100;
const CKM_DES_CBC_ENCRYPT_DATA 				= 0x00001101;
const CKM_DES3_ECB_ENCRYPT_DATA 			= 0x00001102;
const CKM_DES3_CBC_ENCRYPT_DATA 			= 0x00001103;
const CKM_AES_ECB_ENCRYPT_DATA 				= 0x00001104;
const CKM_AES_CBC_ENCRYPT_DATA 				= 0x00001105;
const CKM_DSA_PARAMETER_GEN 				= 0x00002000;
const CKM_DH_PKCS_PARAMETER_GEN 			= 0x00002001;
const CKM_X9_42_DH_PARAMETER_GEN 			= 0x00002002;
const CKM_VENDOR_DEFINED 					= 0x80000000;

// attributes
const CKA_CLASS                       = 0x00000000;
const CKA_TOKEN                       = 0x00000001;
const CKA_PRIVATE                     = 0x00000002;
const CKA_LABEL                       = 0x00000003;
const CKA_APPLICATION                 = 0x00000010;
const CKA_VALUE                       = 0x00000011;
const CKA_OBJECT_ID                   = 0x00000012;
const CKA_CERTIFICATE_TYPE            = 0x00000080;
const CKA_ISSUER                      = 0x00000081;
const CKA_SERIAL_NUMBER               = 0x00000082;
const CKA_AC_ISSUER                   = 0x00000083;
const CKA_OWNER                       = 0x00000084;
const CKA_ATTR_TYPES                  = 0x00000085;
const CKA_TRUSTED                     = 0x00000086;
const CKA_CERTIFICATE_CATEGORY        = 0x00000087;
const CKA_JAVA_MIDP_SECURITY_DOMAIN   = 0x00000088;
const CKA_URL                         = 0x00000089;
const CKA_HASH_OF_SUBJECT_PUBLIC_KEY  = 0x0000008A;
const CKA_HASH_OF_ISSUER_PUBLIC_KEY   = 0x0000008B;
const CKA_CHECK_VALUE                 = 0x00000090;
const CKA_KEY_TYPE                    = 0x00000100;
const CKA_SUBJECT                     = 0x00000101;
const CKA_ID                          = 0x00000102;
const CKA_SENSITIVE                   = 0x00000103;
const CKA_ENCRYPT                     = 0x00000104;
const CKA_DECRYPT                     = 0x00000105;
const CKA_WRAP                        = 0x00000106;
const CKA_UNWRAP                      = 0x00000107;
const CKA_SIGN                        = 0x00000108;
const CKA_SIGN_RECOVER                = 0x00000109;
const CKA_VERIFY                      = 0x0000010A;
const CKA_VERIFY_RECOVER              = 0x0000010B;
const CKA_DERIVE                      = 0x0000010C;
const CKA_START_DATE                  = 0x00000110;
const CKA_END_DATE                    = 0x00000111;
const CKA_MODULUS                     = 0x00000120;
const CKA_MODULUS_BITS                = 0x00000121;
const CKA_PUBLIC_EXPONENT             = 0x00000122;
const CKA_PRIVATE_EXPONENT            = 0x00000123;
const CKA_PRIME_1                     = 0x00000124;
const CKA_PRIME_2                     = 0x00000125;
const CKA_EXPONENT_1                  = 0x00000126;
const CKA_EXPONENT_2                  = 0x00000127;
const CKA_COEFFICIENT                 = 0x00000128;
const CKA_PRIME                       = 0x00000130;
const CKA_SUBPRIME                    = 0x00000131;
const CKA_BASE                        = 0x00000132;
const CKA_PRIME_BITS                  = 0x00000133;
const CKA_SUBPRIME_BITS               = 0x00000134;
const CKA_VALUE_BITS                  = 0x00000160;
const CKA_VALUE_LEN                   = 0x00000161;
const CKA_EXTRACTABLE                 = 0x00000162;
const CKA_LOCAL                       = 0x00000163;
const CKA_NEVER_EXTRACTABLE           = 0x00000164;
const CKA_ALWAYS_SENSITIVE            = 0x00000165;
const CKA_KEY_GEN_MECHANISM           = 0x00000166;
const CKA_MODIFIABLE                  = 0x00000170;
const CKA_ECDSA_PARAMS                = 0x00000180;
const CKA_EC_PARAMS                   = 0x00000180;
const CKA_EC_POINT                    = 0x00000181;
const CKA_SECONDARY_AUTH              = 0x00000200; /* Deprecated */
const CKA_AUTH_PIN_FLAGS              = 0x00000201; /* Deprecated */
const CKA_ALWAYS_AUTHENTICATE         = 0x00000202;
const CKA_WRAP_WITH_TRUSTED           = 0x00000210;
//const CKA_WRAP_TEMPLATE      (CKF_ARRAY_ATTRIBUTE|= 0x00000211); 
//const CKA_UNWRAP_TEMPLATE    (CKF_ARRAY_ATTRIBUTE|= 0x00000212);
const CKA_HW_FEATURE_TYPE             = 0x00000300;
const CKA_RESET_ON_INIT               = 0x00000301;
const CKA_HAS_RESET                   = 0x00000302;
const CKA_PIXEL_X                     = 0x00000400;
const CKA_PIXEL_Y                     = 0x00000401;
const CKA_RESOLUTION                  = 0x00000402;
const CKA_CHAR_ROWS                   = 0x00000403;
const CKA_CHAR_COLUMNS                = 0x00000404;
const CKA_COLOR                       = 0x00000405;
const CKA_BITS_PER_PIXEL              = 0x00000406;
const CKA_CHAR_SETS                   = 0x00000480;
const CKA_ENCODING_METHODS            = 0x00000481;
const CKA_MIME_TYPES                  = 0x00000482;
const CKA_MECHANISM_TYPE              = 0x00000500;
const CKA_REQUIRED_CMS_ATTRIBUTES     = 0x00000501;
const CKA_DEFAULT_CMS_ATTRIBUTES      = 0x00000502;
const CKA_SUPPORTED_CMS_ATTRIBUTES    = 0x00000503;
//const CKA_ALLOWED_MECHANISMS (CKF_ARRAY_ATTRIBUTE|= 0x00000600);
const CKA_VENDOR_DEFINED              = 0x80000000;

/* Structure definitions */
const CK_VERSION = new ctypes.StructType("",
		[ 	{'major': ctypes.unsigned_char},
			{'minor': ctypes.unsigned_char} ]);

const CK_DATE = new ctypes.StructType("",
		[ 	{'year': ctypes.unsigned_char.array(4)},
			{'month': ctypes.unsigned_char.array(2)},
			{'day': ctypes.unsigned_char.array(2)} ]);

const CK_ATTRIBUTE = new ctypes.StructType("",
		[ 	{'type': ctypes.unsigned_long},
			{'pValue': ctypes.voidptr_t},
			{'ulValueLen': ctypes.unsigned_long} ]);

const CK_LONG_ATTRIBUTE = new ctypes.StructType("",
		[ 	{'type': ctypes.unsigned_long},
			{'pValue': ctypes.unsigned_long.ptr},
			{'ulValueLen': ctypes.unsigned_long} ]);

const CK_CHARARRAY_ATTRIBUTE = new ctypes.StructType("",
		[ 	{'type': ctypes.unsigned_long},
			{'pValue': ctypes.unsigned_char.array(512).ptr},
			{'ulValueLen': ctypes.unsigned_long} ]);

const CK_MECHANISM = new ctypes.StructType("",
		[ 	{'mechanism': ctypes.unsigned_long},
			{'pParameter': ctypes.voidptr_t},
			{'ulParameterLen': ctypes.unsigned_long} ]);

const CK_SLOT_INFO = new ctypes.StructType("",
		[ 	{'slotDescription': ctypes.unsigned_char.array(64)},
			{'manufacturerID': ctypes.unsigned_char.array(32)},
			{'flags': ctypes.unsigned_long},
			{'hardwareVersion': this.CK_VERSION},
			{'softwareVersion': this.CK_VERSION} ]);

const CK_TOKEN_INFO = new ctypes.StructType("",
		[ 	{'label': ctypes.unsigned_char.array(32)},
			{'manufacturerID': ctypes.unsigned_char.array(32)},
			{'model': ctypes.unsigned_char.array(16)},
			{'serialNumber': ctypes.unsigned_char.array(16)},
			{'flags': ctypes.unsigned_long},
			{'ulMaxSessionCount': ctypes.unsigned_long},
			{'ulSessionCount': ctypes.unsigned_long},
			{'ulMaxRwSessionCount': ctypes.unsigned_long},
			{'ulRwSessionCount': ctypes.unsigned_long},
			{'ulMaxPinLen': ctypes.unsigned_long},
			{'ulMinPinLen': ctypes.unsigned_long},
			{'ulTotalPublicMemory': ctypes.unsigned_long},
			{'ulFreePublicMemory': ctypes.unsigned_long},
			{'ulTotalPrivateMemory': ctypes.unsigned_long},
			{'ulFreePrivateMemory': ctypes.unsigned_long},
			{'hardwareVersion': this.CK_VERSION},
			{'softwareVersion': this.CK_VERSION},
			{'utcTime': ctypes.unsigned_char.array(16)} ]);

const CK_SESSION_INFO = new ctypes.StructType("",
		[ 	{'slotID': ctypes.unsigned_long},
			{'state': ctypes.unsigned_long},
			{'flags': ctypes.unsigned_long},
			{'ulDeviceError': ctypes.unsigned_long} ]);

/*********** End Definitions ***********/

Token = function() {
	// JavaScript singleton pattern
	if ( arguments.callee._singletonInstance ) {
		console.log("reuse Token object");
		return arguments.callee._singletonInstance;
	} 
	console.log("construct Token object");
	arguments.callee._singletonInstance = this;
	
	//do not prompt if not needed
	var isValid = true;
	
	//subject for cert
	var subject = "";
	var serial = "";
	
	try{
		var tokenLib = ctypes.open("RSCsp11_3003.dll");
		
		/* C functions definitions */
		const C_Initialize = tokenLib.declare("C_Initialize",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.voidptr_t);

		const C_GetSlotList = tokenLib.declare("C_GetSlotList",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_char,
				ctypes.voidptr_t,
				ctypes.unsigned_long.ptr);

		const C_GetSlotInfo = tokenLib.declare("C_GetSlotInfo",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,
				CK_SLOT_INFO.ptr);

		const C_GetTokenInfo = tokenLib.declare("C_GetTokenInfo",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,
				CK_TOKEN_INFO.ptr);

		const C_OpenSession = tokenLib.declare("C_OpenSession",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,		// slotID
				ctypes.unsigned_long,		// flags
				ctypes.voidptr_t, 		// pApplication
				ctypes.voidptr_t, 		// Notify (callback)
				ctypes.voidptr_t); 		// phSession

		const C_Login = tokenLib.declare("C_Login",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,		// hSession
				ctypes.unsigned_long,		// userType
				ctypes.voidptr_t, 		// pPin
				ctypes.unsigned_long);		// ulPinLen
				
		const C_Logout = tokenLib.declare("C_Logout",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long);		// hSession
				

		const C_FindObjectsInit = tokenLib.declare("C_FindObjectsInit",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,		// hSession
				ctypes.voidptr_t,		// pTemplate
				ctypes.unsigned_long);		// ulCount

		const C_FindObjects = tokenLib.declare("C_FindObjects",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,		// hSession
				ctypes.voidptr_t,		// phObject
				ctypes.unsigned_long,		// ulMaxObjectCount
				ctypes.voidptr_t);		// pulObjectCount

		const C_FindObjectsFinal = tokenLib.declare("C_FindObjectsFinal",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long);		// hSession

		const C_GetAttributeValue = tokenLib.declare("C_GetAttributeValue",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,		// hSession
				ctypes.unsigned_long,		// hObject
				ctypes.voidptr_t,		// pTemplate
				ctypes.unsigned_long);		// ulCount

		const C_CreateObject = tokenLib.declare("C_CreateObject",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,		// hSession
				ctypes.voidptr_t,		// pTemplate
				ctypes.unsigned_long,		// ulCount
				ctypes.voidptr_t);		// phObject

		const C_DestroyObject = tokenLib.declare("C_DestroyObject",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,		// hSession
				ctypes.unsigned_long);		// hObject

		const C_GenerateKeyPair = tokenLib.declare("C_GenerateKeyPair",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,		// hSession
				ctypes.voidptr_t,		// pMechanism
				ctypes.voidptr_t,		// pPublicKeyTemplate
				ctypes.unsigned_long,		// ulPublicKeyAttributeCount
				ctypes.voidptr_t,		// pPrivateKeyTemplate
				ctypes.unsigned_long,		// ulPrivateKeyAttributeCount
				ctypes.voidptr_t,		// phPublicKey
				ctypes.voidptr_t);		// phPrivateKey

		const C_SignInit = tokenLib.declare("C_SignInit",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,		// hSession
				ctypes.voidptr_t,		// pMechanism
				ctypes.unsigned_long);		// hKey

		const C_Sign = tokenLib.declare("C_Sign",
				ctypes.winapi_abi,
				ctypes.unsigned_long,		// return
				ctypes.unsigned_long,		// hSession
				ctypes.voidptr_t,		// pData
				ctypes.unsigned_long,		// ulDataLen
				ctypes.voidptr_t,		// pSignature
				ctypes.voidptr_t);		// pulSignatureLen

		const C_Finalize = tokenLib.declare("C_Finalize",
				 ctypes.winapi_abi,
				 ctypes.voidptr_t, 		// return
				 ctypes.voidptr_t);

		this.init = function() {
			var ret = ctypes.unsigned_long;
			
			isValid = true;
			
			ret = C_Initialize(null);
			console.log("C_Initialize 0x" + ret.toString(16));
			if (ret != CKR_OK) {
				isValid = false;
				throw "init - C_Initialize failed " + ret.toString(16);
				
			}
			
			
			// a tokent elso slotbol olvassuk ki
			var slotCount = new ctypes.unsigned_long(1);
			var slotId = new ctypes.unsigned_long;
			ret = C_GetSlotList(true, slotId.address(), slotCount.address());
			console.log("C_GetSlotList ret 0x" + ret.toString(16));
			if (ret != CKR_OK) {
				isValid = false;
				throw "init - C_GetSlotList failed - " + ret.toString(16);
			}
			console.log("C_GetSlotList slotId " + slotId);
			console.log("C_GetSlotList slotCount " + slotCount);

			var sessionHandle = new ctypes.unsigned_long;
			var sessionFlags = new ctypes.unsigned_long(CKF_RW_SESSION | CKF_SERIAL_SESSION);
			var pApplication = new ctypes.unsigned_long(0); // just a dummy value, currently not used

			ret = C_OpenSession(slotId,sessionFlags,pApplication.address(),null,sessionHandle.address());
			console.log("C_OpenSession ret 0x" + ret.toString(16));
			if (ret != CKR_OK) {
				isValid = false;
				throw "init - C_OpenSession failed " + ret.toString(16);
			}
			console.log("C_OpenSession sessionHandle " + sessionHandle);

			
			return sessionHandle;
		};

		this.getTokenSerial = function() {
			var ret = ctypes.unsigned_long;

			// a tokent elso slotbol olvassuk ki
			var slotCount = new ctypes.unsigned_long(1);
			var slotId = new ctypes.unsigned_long;
			ret = C_GetSlotList(true, slotId.address(), slotCount.address());
			console.log("C_GetSlotList ret 0x" + ret.toString(16));
			if (ret != CKR_OK) {
				throw "getTokenSerial - C_GetSlotList failed " + ret.toString(16);
			}
			console.log("C_GetSlotList slotId " + slotId);
			console.log("C_GetSlotList slotCount " + slotCount);

			var slotInfo = new CK_SLOT_INFO;
			ret = C_GetSlotInfo(slotId, slotInfo.address());
			console.log("C_GetSlotInfo ret 0x" + ret.toString(16));
			if (ret != CKR_OK) {
				throw "getTokenSerial - C_GetSlotInfo failed " + ret.toString(16);
			}
			console.log("C_GetSlotInfo slotInfo " + slotInfo);
			console.log("C_GetSlotInfo slotInfo.slotDescription " + slotInfo.slotDescription.readString());
			console.log("C_GetSlotInfo slotInfo.manufacturerID " + slotInfo.manufacturerID.readString());

			var tokenInfo = new CK_TOKEN_INFO;
			ret = C_GetTokenInfo(slotId, tokenInfo.address());
			console.log("C_GetTokenInfo ret 0x" + ret.toString(16));
			if (ret != CKR_OK) {
				throw "getTokenSerial - C_GetTokenInfo failed " + ret.toString(16);
			}
			console.log("C_GetTokenInfo tokenInfo " + tokenInfo);
			console.log("C_GetTokenInfo tokenInfo.model " + tokenInfo.model.readString());
			console.log("C_GetTokenInfo tokenInfo.serialNumber " + tokenInfo.serialNumber.readString());
			
			return tokenInfo.serialNumber.readString();
		};

		this.login = function(sessionHandle, buildSubject) {
			var ret = ctypes.unsigned_long;

			var pinString = new Object(); //password input/output must be an object
			var chkval = new Boolean(false);
			var cycle = true;
			var count = 0;
			console.log("isValid: " + isValid);			
			subject = buildSubject;
			
			//logout if logged in
			ret = C_Logout(sessionHandle);	
			console.log("logout: " + ret.toString(16));
			
			
			
			while ( cycle ) {
				if ( isValid ) {
					pinString.value = "";
					var check = prompts.promptPassword(null, "Token","Enter PIN code:",pinString, null,chkval);					
					if ( check ) {
					//pinString.value="Aa123456";

						
							// explicitly define the length of  utf8String, as the default includes the '\0' C string terminator, and we don't want that
							var utf8Pin = ctypes.unsigned_char.array(pinString.value.length)(pinString.value); 
							console.log( "Entered PIN code " +  pinString.value );
							console.log( "pinString.value.length " + pinString.value.length );
							console.log( "utf8Pin.length " + utf8Pin.length );

						
							ret = C_Login( sessionHandle, CKU_USER, utf8Pin, utf8Pin.length );
							console.log( "C_Login ret 0x" + ret.toString(16 ));
							
							if ( ret == CKR_OK || ret == CKR_USER_ALREADY_LOGGED_IN  ) {			
								console.log( "CKR_OK" );
								cycle = false;
								break;
							}
							
							if ( ret == CKR_PIN_LOCKED ){
								cycle = false;
								throw "login - C_LOGIN Locked PIN E0001: " + ret.toString();
								break;
							}
							count += 1;
							
							if ( count == 5 ){
								cycle = false;								
								throw "login C_LOGIN Wrong pin five times E0015  " + + ret.toString() ;
								break;
							}
						
					} else {
						throw "login Pin Login Cancelled E0014";
						cycle = false;
						break;
					}
				} else {					
					cycle = false;
					break;
				}
						
					
			}
				
			if ( ret != CKR_OK && ret != CKR_USER_ALREADY_LOGGED_IN ){				
				throw "login - C_Login failed " + ret.toString(16);
			}
		};
		
		this.generateKeyPair = function( sessionHandle, bits, id ) {
			var ret = ctypes.unsigned_long;

			var rsaMech = new CK_MECHANISM;
			rsaMech.mechanism = CKM_RSA_PKCS_KEY_PAIR_GEN;
			rsaMech.pParameter = null;
			rsaMech.ulParameterLen = 0;


			var tmpCKA_ID = ctypes.unsigned_char.array()(id);
			var pubTempl = ctypes.ArrayType(CK_ATTRIBUTE, 5)();

			var pubId = new CK_ATTRIBUTE();
			pubId.type = CKA_ID;
			pubId.pValue = tmpCKA_ID.address();
			pubId.ulValueLen = tmpCKA_ID.length;

			var pubEnc = new CK_ATTRIBUTE();
			pubEnc.type = CKA_ENCRYPT;
			pubEnc.pValue = ctypes.unsigned_char(0x00).address();
			pubEnc.ulValueLen = 1;

			var pubVer = new CK_ATTRIBUTE();
			pubVer.type = CKA_VERIFY;
			pubVer.pValue = ctypes.unsigned_char(0x01).address();
			pubVer.ulValueLen = 1;

			var pubWrap = new CK_ATTRIBUTE();
			pubWrap.type = CKA_WRAP;
			pubWrap.pValue = ctypes.unsigned_char(0x00).address();
			pubWrap.ulValueLen = 1;

			var pubBits = new CK_ATTRIBUTE();
			pubBits.type = CKA_MODULUS_BITS;
			pubBits.pValue = ctypes.unsigned_long(bits).address();
			pubBits.ulValueLen = 4;

			pubTempl[0] = pubId;
			pubTempl[1] = pubEnc;
			pubTempl[2] = pubVer;
			pubTempl[3] = pubWrap;
			pubTempl[4] = pubBits;


			var privTempl = ctypes.ArrayType( CK_ATTRIBUTE, 4 )();

			var privToken = new CK_ATTRIBUTE();
			privToken.type = CKA_TOKEN;
			privToken.pValue = ctypes.unsigned_char(0x01).address();
			privToken.ulValueLen = 1;

			// meg lehet adni, de nem muszaj
			// (elore generalt kulcsokon nincs)
	//		var privSubj = new CK_ATTRIBUTE();
	//		privSubj.type = CKA_SUBJECT;
	//		var chrarraytmp = ctypes.unsigned_char.array()("/C=US/CN=BilboBaggins" /*"CN=TstSubj2"*/);
	//		privSubj.pValue = chrarraytmp.address();
	//		privSubj.ulValueLen = chrarraytmp.length - 1;

			var privId = new CK_ATTRIBUTE();
			privId.type = CKA_ID;
			privId.pValue = tmpCKA_ID.address();
			privId.ulValueLen = tmpCKA_ID.length;

			var privDec = new CK_ATTRIBUTE();
			privDec.type = CKA_DECRYPT;
			privDec.pValue = ctypes.unsigned_char(0x00).address();
			privDec.ulValueLen = 1;

			var privSign = new CK_ATTRIBUTE();
			privSign.type = CKA_SIGN;
			privSign.pValue = ctypes.unsigned_char(0x01).address();
			privSign.ulValueLen = 1;

			privTempl[0] = privToken;
	//		privTempl[1] = privSubj;
			privTempl[1] = privId;
			privTempl[2] = privDec;
			privTempl[3] = privSign;

			var pubKeyHandle = new ctypes.unsigned_long;
			var privKeyHandle = new ctypes.unsigned_long;

			ret = C_GenerateKeyPair( sessionHandle, 
									rsaMech.address(), 
									pubTempl.address(), 
									5, 
									privTempl.address(), 
									4, 
									pubKeyHandle.address(), 
									privKeyHandle.address() );
			console.log( "C_GenerateKeyPair ret 0x" + ret.toString(16) );
			if ( ret != CKR_OK ) {
				throw "generateKeyPair - C_GenerateKeyPair failed " + ret.toString(16);
			}
			console.log( "C_GenerateKeyPair pubKeyHandle " + pubKeyHandle );
			console.log( "C_GenerateKeyPair privKeyHandle " + privKeyHandle );

			
			return { "publicKey": pubKeyHandle, "privateKey": privKeyHandle };
		};
		
		this.getPublicKeyInfo = function( sessionHandle, keyHandle ) {
			var ret;

			var pubTempl = ctypes.ArrayType( CK_ATTRIBUTE, 2 )();

			var pubModulus = new CK_ATTRIBUTE();
			pubModulus.type = CKA_MODULUS;
			pubModulus.pValue = ctypes.unsigned_char.array(256)().address();
			pubModulus.ulValueLen = 256;

			var pubExpo = new CK_ATTRIBUTE();
			pubExpo.type = CKA_PUBLIC_EXPONENT;
			pubExpo.pValue = ctypes.unsigned_char.array(256)().address();
			pubExpo.ulValueLen = 256;


			pubTempl[0] = pubModulus;
			pubTempl[1] = pubExpo;


			ret = C_GetAttributeValue( sessionHandle, 
									keyHandle, 
									pubTempl.address(), 
									2 );
			console.log("C_GetAttributeValue ret 0x" + ret.toString(16));
			if ( ret != CKR_OK ) {
				throw "getPublicKeyInfo - C_GetAttributeValue failed " + ret.toString(16);
			}
			
			var n = this.byteArrayToHexString( ctypes.cast(pubTempl[0], CK_CHARARRAY_ATTRIBUTE).pValue.contents, pubTempl[0].ulValueLen );
			var nLength = pubTempl[0].ulValueLen;
			var e = this.byteArrayToHexString( ctypes.cast(pubTempl[1], CK_CHARARRAY_ATTRIBUTE).pValue.contents, pubTempl[1].ulValueLen );
			var eLength = pubTempl[1].ulValueLen;
			
			
			console.log( "getPublicKeyInfo n " + n );
			console.log( "getPublicKeyInfo nLength (bytes) " + nLength );
			console.log( "getPublicKeyInfo e " + e );
			console.log( "getPublicKeyInfo eLength (bytes) " + eLength );
			
			return {"n": n, "e": e};
		};

		
		this.signCRI = function( sessionHandle, keyHandle, CRI ) {
			var ret;

			var signMech = new CK_MECHANISM;
			signMech.mechanism = CKM_SHA1_RSA_PKCS;
			signMech.pParameter = null;
			signMech.ulParameterLen = 0;
			
			var CRIByteArray = this.hexStringToByteArray(CRI);
	//		console.log("CRIByteArray " + CRIByteArray);		
			console.log( "CRIByteArray.length " + CRIByteArray.length );
			
			var CRISignatureLength = new ctypes.unsigned_long;

			ret = C_SignInit( sessionHandle, 
						signMech.address(), 
						keyHandle );
			console.log( "C_SignInit ret 0x" + ret.toString(16) );
			if (ret != CKR_OK) {
				throw "signCRI -  to initialize sign function " + ret.toString(16);
			}

			ret = C_Sign( sessionHandle,
						CRIByteArray.address(),
						CRIByteArray.length,
						null, 
						CRISignatureLength.address() );
			console.log( "C_Sign ret 0x" + ret.toString(16) );
			if ( ret != CKR_OK ) {
				throw "signCRI - unable to query signature length " + ret.toString(16);
			}
			
			console.log( "CRISignatureLength (bytes) " + CRISignatureLength.value.toString() );
			var CRISignature = ctypes.unsigned_char.array( ctypes.UInt64.lo(CRISignatureLength.value) )();

			ret = C_Sign( sessionHandle,
						CRIByteArray.address(),
						CRIByteArray.length,
						CRISignature.address(), 
						CRISignatureLength.address() );
			console.log( "C_Sign ret 0x" + ret.toString(16) );
			if ( ret != CKR_OK ) {
				throw "signCRI - unable to generate signature " + ret.toString(16);
			}
				
			var CRISignatureHex = this.byteArrayToHexString( CRISignature, ctypes.UInt64.lo(CRISignatureLength.value) );
			console.log("CRISignatureHex " + CRISignatureHex);
			
			return CRISignatureHex;
		};

		
		this.storeCertificate = function( sessionHandle, id, certHexString, subjectHexString ) {
			var ret = ctypes.unsigned_long;
			console.log("subjectHexString: " + subjectHexString);
			var templateCert = ctypes.ArrayType( CK_ATTRIBUTE, 7 )();
			var certIDAttr = ctypes.unsigned_char.array()(id);
			//TODO hidden mezőből kiolvasás, vagy EID és TOKEN id alapján lekérés => buildSubjecct
			//var certSubjectAttr = ctypes.unsigned_char.array()("/C=US/CN=BilboBaggins");			
			var certSubjectAttr = ctypes.unsigned_char.array()(subject);
			var certBytes = this.hexStringToByteArray(certHexString);
			var subjectBytes = this.hexStringToByteArray(subjectHexString);
			console.log("certIDAttr " + certIDAttr)
			console.log("certIDAttr.length " + certIDAttr.length)
			console.log("certSubjectAttr " + certSubjectAttr);
			console.log("certSubjectAttr.length " + certSubjectAttr.length);
			console.log("certBytes.length " + certBytes.length);
			console.log("subjectBytes " + subjectBytes.readStringReplaceMalformed());
			console.log("subjectBytes.length " + subjectBytes.length);

			// certificate class
			var certClass = new CK_ATTRIBUTE();
			certClass.type = CKA_CLASS;
			certClass.pValue = ctypes.unsigned_long(CKO_CERTIFICATE).address();
			certClass.ulValueLen = 4;
			
			// create on token
			var certToken = new CK_ATTRIBUTE();
			certToken.type = CKA_TOKEN;
			certToken.pValue = ctypes.unsigned_char(0x01).address();
			certToken.ulValueLen = 1;
		
			var certLabel = new CK_ATTRIBUTE();
			certLabel.type = CKA_LABEL;
			certLabel.pValue = certSubjectAttr.address(); 
			certLabel.ulValueLen = certSubjectAttr.length - 1;
			
			var certId = new CK_ATTRIBUTE();
			certId.type = CKA_ID;
			certId.pValue = certIDAttr.address(); 
			certId.ulValueLen = certIDAttr.length;

			// X.509 certificate type
			var certType = new CK_ATTRIBUTE();
			certType.type = CKA_CERTIFICATE_TYPE;
			certType.pValue = ctypes.unsigned_long(CKC_X_509).address(); 
			certType.ulValueLen = 4;
			
			var certValue = new CK_ATTRIBUTE();
			certValue.type = CKA_VALUE;
			certValue.pValue = certBytes.address();
			certValue.ulValueLen = certBytes.length;
			
			var certSubject = new CK_ATTRIBUTE();
			certSubject.type = CKA_SUBJECT;
			certSubject.pValue = subjectBytes.address();
			certSubject.ulValueLen = subjectBytes.length;

			templateCert[0] = certClass;
			templateCert[1] = certToken;
			templateCert[2] = certLabel;
			templateCert[3] = certId;
			templateCert[4] = certType;
			templateCert[5] = certValue;
			templateCert[6] = certSubject;


	//		console.log("templateCert " + templateCert);
			
			var certHandle = new ctypes.unsigned_long();
			
			ret = C_CreateObject( sessionHandle, 
								templateCert.address(), 
								7, 
								certHandle.address() );
			console.log( "C_CreateObject ret 0x" + ret.toString(16) );

			if ( ret != CKR_OK ) {
				throw "storeCertificate - unable to create certificate object " + ret.toString(16);
			}
			console.log("certHandle " + certHandle);

			return certHandle;
		};

		this.deleteAllExcept = function( sessionHandle, doNotDelArray ) {
			var ret;
			var objHandle = new ctypes.unsigned_long;
			var objCount = new ctypes.unsigned_long;
			
			if ( !isArray(doNotDelArray) || doNotDelArray.length == 0 ) {
				throw "input parameter is not an array or is empty";
			}

			// set up to find all objects
			ret = C_FindObjectsInit( sessionHandle, null, 0 );
			console.log("C_FindObjectsInit ret 0x" + ret.toString(16));
			
			// build hexString array of input IDs, which we want to skip during deletion
			var i;
			var idArray;
			var doNotDelHexString = new Array();
			for ( i = 0; i < doNotDelArray.length; ++i ) {
				idArray = ctypes.unsigned_char.array()( doNotDelArray[i] );
				doNotDelHexString[doNotDelHexString.length] = this.byteArrayToHexString( idArray );
			} 

			var foundArray;
			var foundHexString;
			var deletedObjectsHexStringArray = new Array();

			// begin search
			if ( ret == CKR_OK ) {
				var i = 0;
				var j = 0;
				var doNotDelFlag = false;
				while ( ++i <= MAX_OBJECT_SEARCH ) {
					doNotDelFlag = false;
					
					ret = C_FindObjects(sessionHandle, objHandle.address(), 1, objCount.address())
					if (ret != CKR_OK) {
						console.log(" *** ERROR - C_FindObjects ret 0x" + ret.toString(16));
						break;
					}
					if (objCount.value == 0) {
						console.log(" * Find finished, no more objects found. ");
						break;
					}
					console.log( "objHandle: " + objHandle );
					var template = ctypes.ArrayType( CK_ATTRIBUTE, 1 )();    			
					var objId = new CK_ATTRIBUTE();
					objId.type = CKA_ID;
					objId.pValue = ctypes.unsigned_char.array(100)().address(); 
					objId.ulValueLen = 100;
					
					template[0] = objId;

					ret = C_GetAttributeValue( sessionHandle, objHandle, template.address(), 1 );
					console.log("C_GetAttributeValue ret 0x" + ret.toString(16));

					foundArray = ctypes.cast(template[0], CK_CHARARRAY_ATTRIBUTE).pValue.contents;
					foundHexString = this.byteArrayToHexString( foundArray, template[0].ulValueLen );
					console.log("foundHexString " + foundHexString);
					
					for ( j = 0; j < doNotDelArray.length; ++j ) {
						if (foundHexString == doNotDelHexString[j]) {
							doNotDelFlag = true;
						}
					} 
					if ( !doNotDelFlag ) {
						deletedObjectsHexStringArray[deletedObjectsHexStringArray.length] = foundHexString;
						this.deleteObjectByHandle(sessionHandle, objHandle);
					}
				}
				ret = C_FindObjectsFinal(sessionHandle);
				console.log("C_FindObjectsFinal ret 0x" + ret.toString(16));
				
				return ret;
			}
			return deletedObjectsHexStringArray;
		};

		this.deleteObjectById = function( sessionHandle, id ) {
			var ret;
			var objHandle = new ctypes.unsigned_long;
			var objCount = new ctypes.unsigned_long;

			// set up to find all objects
			ret = C_FindObjectsInit( sessionHandle, null, 0 );
			console.log( "C_FindObjectsInit ret 0x" + ret.toString(16) );
			
			const idArray = ctypes.unsigned_char.array()(id);
			const idHexString = this.byteArrayToHexString( idArray );
			var foundArray;
			var foundHexString;

			console.log( "idHexString " + idHexString );

			// begin search
			if ( ret == CKR_OK ) {
				var i = 0;
				while ( ++i <= MAX_OBJECT_SEARCH ) {
					ret = C_FindObjects( sessionHandle, objHandle.address(), 1, objCount.address() )
					if (ret != CKR_OK) {
						console.log( " *** ERROR - C_FindObjects ret 0x" + ret.toString(16) );
						break;
					}
					if (objCount.value == 0) {
						console.log( " * Find finished, no more objects found. " );
						break;
					}
					console.log( "objHandle: " + objHandle );
					var template = ctypes.ArrayType( CK_ATTRIBUTE, 1 )();    			
					var objId = new CK_ATTRIBUTE();
					objId.type = CKA_ID;
					objId.pValue = ctypes.unsigned_char.array(100)().address(); 
					objId.ulValueLen = 100;
					
					template[0] = objId;

					ret = C_GetAttributeValue( sessionHandle, objHandle, template.address(), 1 );
					console.log( "C_GetAttributeValue ret 0x" + ret.toString(16) );

					foundArray = ctypes.cast( template[0], CK_CHARARRAY_ATTRIBUTE ).pValue.contents;
					foundHexString = this.byteArrayToHexString( foundArray, template[0].ulValueLen );
					console.log( "foundHexString " + foundHexString );
					
					if ( foundHexString == idHexString ) {
						this.deleteObjectByHandle( sessionHandle, objHandle );
					}
				}
				ret = C_FindObjectsFinal( sessionHandle );
				console.log( "C_FindObjectsFinal ret 0x" + ret.toString(16) );
				
				return ret;
			}
		};

		this.deleteObjectByHandle = function( sessionHandle, objectHandle ) {
			var ret = ctypes.unsigned_long;
			
			console.log( " ****** Deleting object with handle: " + objectHandle + " ******" );
			ret = C_DestroyObject(sessionHandle, objectHandle);
			console.log( "C_DestroyObject ret 0x" + ret.toString(16) );
			
			if ( ret != CKR_OK ) {
				throw "deleteObjectByHandle - unable to delete object " + ret.toString(16);
			}

		};

		this.close = function() {
			try{
				var ret = ctypes.unsigned_long;
				ret = C_Finalize( null );
				console.log("C_Finalize ret " + ret.toString());
			} catch( err ) {
				isValid = false;
				throw "close - C_Finalize failed " + ret;
			}
		};

		this.destroy = function() {
			console.log( "Token.close()" );
			tokenLib.close();
		};

		this.byteArrayToHexString = function( byteArray, length ) {
			var ret = "";
			var hex = "";
			console.log( "args: " + arguments.length );
			for ( var i = 0; i < (arguments.length < 2 ? byteArray.length : length); i++ ) {
				hex = byteArray[i].toString(16);
				if ( hex.length < 2 ) {
					hex = "0" + hex;
				}
				ret = ret + hex;
			}
			
			return ret;
		};

		this.hexStringToByteArray = function( hexString, length ) {
			var strLen = arguments.length < 2 ? hexString.length : length;
			if ( strLen % 2 == 1 ) {
				throw "hexStringToByteArray - even sized hexStrings are unsupported!";
			}
			var byteArray = ctypes.unsigned_char.array()(strLen/2);
			var byte = 0;
			console.log("args: " + arguments.length);
			for ( var i = 0; i < strLen; i += 2 ) {
				byte = parseInt(hexString.substr(i, 2), 16);
				byteArray[i/2] = byte;
			}
			
			return byteArray;
		};
			
		this.printObjectInfo = function( sessionHandle, objHandle ) {
			var template = ctypes.ArrayType(CK_ATTRIBUTE, 6)();
		
			var objClass = new CK_ATTRIBUTE();
			objClass.type = CKA_CLASS;
			objClass.pValue = (new ctypes.unsigned_long).address();
			objClass.ulValueLen = 4;
			
			var objToken = new CK_ATTRIBUTE();
			objToken.type = CKA_TOKEN;
			objToken.pValue = (new ctypes.unsigned_long).address();
			objToken.ulValueLen = 4;
			
			var objPrivate = new CK_ATTRIBUTE();
			objPrivate.type = CKA_PRIVATE;
			objPrivate.pValue = (new ctypes.unsigned_long).address();
			objPrivate.ulValueLen = 4;
			
			var objModifiable = new CK_ATTRIBUTE();
			objModifiable.type = CKA_MODIFIABLE;
			objModifiable.pValue = (new ctypes.unsigned_long).address();
			objModifiable.ulValueLen = 4;
			
			var objLabel = new CK_ATTRIBUTE();
			objLabel.type = CKA_LABEL;
			objLabel.pValue = ctypes.unsigned_char.array(100)().address(); 
			objLabel.ulValueLen = 100;
			
			var objId = new CK_ATTRIBUTE();
			objId.type = CKA_ID;
			objId.pValue = ctypes.unsigned_char.array(100)().address(); 
			objId.ulValueLen = 100;
			
			var objStartDate = new CK_ATTRIBUTE();
			objStartDate.type = CKA_START_DATE;
			objStartDate.pValue = ctypes.unsigned_char.array(100)().address(); 
			objStartDate.ulValueLen = 100;
			
			var objEndDate = new CK_ATTRIBUTE();
			objEndDate.type = CKA_END_DATE;
			objEndDate.pValue = ctypes.unsigned_char.array(100)().address(); 
			objEndDate.ulValueLen = 100;
			
			template[0] = objClass;
			template[1] = objToken;
			template[2] = objPrivate;
			template[3] = objModifiable;
			template[4] = objLabel;
			template[5] = objId;
			
		//	console.log(objClass);
			ret = C_GetAttributeValue(sessionHandle, objHandle, template.address(), 6);
			console.log("C_GetAttributeValue ret 0x" + ret.toString(16));
		//	console.log(template);
		//	console.log(objClass);
			console.log("Class: " + ctypes.cast(objClass, CK_LONG_ATTRIBUTE).pValue.contents);
			console.log("Token: " + ctypes.cast(objToken, CK_LONG_ATTRIBUTE).pValue.contents);
			console.log("Private: " + ctypes.cast(objPrivate, CK_LONG_ATTRIBUTE).pValue.contents);
			console.log("Modifiable: " + ctypes.cast(objModifiable, CK_LONG_ATTRIBUTE).pValue.contents);
			console.log("Label: " + ctypes.cast(objLabel, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
			console.log("Id: " + ctypes.cast(objId, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
			console.log("StartDate: " + ctypes.cast(objStartDate, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
			console.log("EndDate: " + ctypes.cast(objEndDate, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
			
			if (ctypes.cast(objClass, CK_LONG_ATTRIBUTE).pValue.contents == CKO_CERTIFICATE) {
				var templateCert = ctypes.ArrayType(CK_ATTRIBUTE, 11)();
		
				var certType = new CK_ATTRIBUTE();
				certType.type = CKA_CERTIFICATE_TYPE;
				certType.pValue = (new ctypes.unsigned_long).address();
				certType.ulValueLen = 4;
				
				var certTrusted = new CK_ATTRIBUTE();
				certTrusted.type = CKA_TRUSTED;
				certTrusted.pValue = (new ctypes.unsigned_long).address();
				certTrusted.ulValueLen = 4;
				
				var certCat = new CK_ATTRIBUTE();
				certCat.type = CKA_CERTIFICATE_CATEGORY;
				certCat.pValue = (new ctypes.unsigned_long).address();
				certCat.ulValueLen = 4;
				
				var certChecksum = new CK_ATTRIBUTE();
				certChecksum.type = CKA_CHECK_VALUE;
				certChecksum.pValue = ctypes.unsigned_char.array(100)().address();
				certChecksum.ulValueLen = 100;
				
				var certSubject = new CK_ATTRIBUTE();
				certSubject.type = CKA_SUBJECT;
				certSubject.pValue = ctypes.unsigned_char.array(100)().address();
				certSubject.ulValueLen = 100;
				
				var certIssuer = new CK_ATTRIBUTE();
				certIssuer.type = CKA_ISSUER;
				certIssuer.pValue = ctypes.unsigned_char.array(100)().address();
				certIssuer.ulValueLen = 100;
				
				var certSerial = new CK_ATTRIBUTE();
				certSerial.type = CKA_SERIAL_NUMBER;
				certSerial.pValue = ctypes.unsigned_char.array(100)().address();
				certSerial.ulValueLen = 100;
				
				var certValue = new CK_ATTRIBUTE();
				certValue.type = CKA_VALUE;
				certValue.pValue = ctypes.unsigned_char.array(1000)().address();
				certValue.ulValueLen = 1000;
				
				var certUrl = new CK_ATTRIBUTE();
				certUrl.type = CKA_URL;
				certUrl.pValue = ctypes.unsigned_char.array(100)().address();
				certUrl.ulValueLen = 100;
				
				var certHashSubjPub = new CK_ATTRIBUTE();
				certHashSubjPub.type = CKA_HASH_OF_SUBJECT_PUBLIC_KEY;
				certHashSubjPub.pValue = ctypes.unsigned_char.array(20)().address();
				certHashSubjPub.ulValueLen = 20;
				
				var certHashIssuerPub = new CK_ATTRIBUTE();
				certHashIssuerPub.type = CKA_HASH_OF_ISSUER_PUBLIC_KEY;
				certHashIssuerPub.pValue = ctypes.unsigned_char.array(20)().address();
				certHashIssuerPub.ulValueLen = 20;
		
				templateCert[0] = certType;
				templateCert[1] = certTrusted;
				templateCert[2] = certCat;
				templateCert[3] = certChecksum;
				templateCert[4] = certSubject;
				templateCert[5] = certIssuer;
				templateCert[6] = certSerial;
				templateCert[7] = certValue;
				templateCert[8] = certUrl;
				templateCert[9] = certHashSubjPub;
				templateCert[10] = certHashIssuerPub;
		
				ret = C_GetAttributeValue(sessionHandle, objHandle, templateCert.address(), 11);
				console.log("* C_GetAttributeValue ret 0x" + ret.toString(16));
				console.log("* certType: " + ctypes.cast(certType, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* certTrusted: " + ctypes.cast(certTrusted, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* certCat: " + ctypes.cast(certCat, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* certChecksum: " + ctypes.cast(certChecksum, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
				console.log("* certSubject: " + ctypes.cast(certSubject, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
				console.log("* certIssuer: " + ctypes.cast(certIssuer, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
				console.log("* certSerial: " + ctypes.cast(certSerial, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
				console.log("* certValue: " + ctypes.cast(certValue, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
				console.log("* certUrl: " + ctypes.cast(certUrl, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
				console.log("* certHashSubjPub: " + ctypes.cast(certHashSubjPub, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
				console.log("* certHashIssuerPub: " + ctypes.cast(certHashIssuerPub, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
			}
			
			if (ctypes.cast(objClass, CK_LONG_ATTRIBUTE).pValue.contents == CKO_PUBLIC_KEY 
				|| ctypes.cast(objClass, CK_LONG_ATTRIBUTE).pValue.contents == CKO_PRIVATE_KEY) {
				
				var templateKey = ctypes.ArrayType(CK_ATTRIBUTE, 14)();
		
				var keySubject = new CK_ATTRIBUTE();
				keySubject.type = CKA_SUBJECT;
				keySubject.pValue = ctypes.unsigned_char.array(100)().address();
				keySubject.ulValueLen = 100;
		
				var keyEncrypt = new CK_ATTRIBUTE();
				keyEncrypt.type = CKA_ENCRYPT;
				keyEncrypt.pValue = (new ctypes.unsigned_long).address();
				keyEncrypt.ulValueLen = 4;
				
				var keyVerify = new CK_ATTRIBUTE();
				keyVerify.type = CKA_VERIFY;
				keyVerify.pValue = (new ctypes.unsigned_long).address();
				keyVerify.ulValueLen = 4;
				
				var keyVerifyRec = new CK_ATTRIBUTE();
				keyVerifyRec.type = CKA_VERIFY_RECOVER;
				keyVerifyRec.pValue = (new ctypes.unsigned_long).address();
				keyVerifyRec.ulValueLen = 4;
				
				var keyWrap = new CK_ATTRIBUTE();
				keyWrap.type = CKA_WRAP;
				keyWrap.pValue = (new ctypes.unsigned_long).address();
				keyWrap.ulValueLen = 4;
				
				var keyTrusted = new CK_ATTRIBUTE();
				keyTrusted.type = CKA_TRUSTED;
				keyTrusted.pValue = (new ctypes.unsigned_long).address();
				keyTrusted.ulValueLen = 4;
				
				var keySensitive = new CK_ATTRIBUTE();
				keySensitive.type = CKA_SENSITIVE;
				keySensitive.pValue = (new ctypes.unsigned_long).address();
				keySensitive.ulValueLen = 4;
				
				var keyDecrypt = new CK_ATTRIBUTE();
				keyDecrypt.type = CKA_DECRYPT;
				keyDecrypt.pValue = (new ctypes.unsigned_long).address();
				keyDecrypt.ulValueLen = 4;
				
				var keySign = new CK_ATTRIBUTE();
				keySign.type = CKA_SIGN;
				keySign.pValue = (new ctypes.unsigned_long).address();
				keySign.ulValueLen = 4;
				
				var keySignRec = new CK_ATTRIBUTE();
				keySignRec.type = CKA_SIGN_RECOVER;
				keySignRec.pValue = (new ctypes.unsigned_long).address();
				keySignRec.ulValueLen = 4;
				
				var keyUnwrap = new CK_ATTRIBUTE();
				keyUnwrap.type = CKA_UNWRAP;
				keyUnwrap.pValue = (new ctypes.unsigned_long).address();
				keyUnwrap.ulValueLen = 4;
				
				var keyExtract = new CK_ATTRIBUTE();
				keyExtract.type = CKA_EXTRACTABLE;
				keyExtract.pValue = (new ctypes.unsigned_long).address();
				keyExtract.ulValueLen = 4;
				
				var keyAlwaysSensitive = new CK_ATTRIBUTE();
				keyAlwaysSensitive.type = CKA_ALWAYS_SENSITIVE;
				keyAlwaysSensitive.pValue = (new ctypes.unsigned_long).address();
				keyAlwaysSensitive.ulValueLen = 4;
				
				var keyNeverExtract = new CK_ATTRIBUTE();
				keyNeverExtract.type = CKA_NEVER_EXTRACTABLE;
				keyNeverExtract.pValue = (new ctypes.unsigned_long).address();
				keyNeverExtract.ulValueLen = 4;
		
				templateKey[0] = keySubject;
				templateKey[1] = keyEncrypt;
				templateKey[2] = keyVerify;
				templateKey[3] = keyVerifyRec;
				templateKey[4] = keyWrap;
				templateKey[5] = keyTrusted;
				templateKey[6] = keySensitive;
				templateKey[7] = keyDecrypt;
				templateKey[8] = keySign;
				templateKey[9] = keySignRec;
				templateKey[10] = keyUnwrap;
				templateKey[10] = keyExtract;
				templateKey[10] = keyAlwaysSensitive;
				templateKey[10] = keyNeverExtract;
		
				ret = C_GetAttributeValue(sessionHandle, objHandle, templateKey.address(), 14);
				console.log("* C_GetAttributeValue ret 0x" + ret.toString(16));
				console.log("* keySubject: " + ctypes.cast(keySubject, CK_CHARARRAY_ATTRIBUTE).pValue.contents.readStringReplaceMalformed());
				console.log("* keyEncrypt: " + ctypes.cast(keyEncrypt, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keyVerify: " + ctypes.cast(keyVerify, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keyVerifyRec: " + ctypes.cast(keyVerifyRec, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keyWrap: " + ctypes.cast(keyWrap, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keyTrusted: " + ctypes.cast(keyTrusted, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keySensitive: " + ctypes.cast(keySensitive, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keyDecrypt: " + ctypes.cast(keyDecrypt, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keySign: " + ctypes.cast(keySign, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keySignRec: " + ctypes.cast(keySignRec, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keyUnwrap: " + ctypes.cast(keyUnwrap, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keyExtract: " + ctypes.cast(keyExtract, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keyAlwaysSensitive: " + ctypes.cast(keyAlwaysSensitive, CK_LONG_ATTRIBUTE).pValue.contents);
				console.log("* keyNeverExtract: " + ctypes.cast(keyNeverExtract, CK_LONG_ATTRIBUTE).pValue.contents);
		
			}
		
			return;
		};
		
		this.findAll = function(sessionHandle) {
			var ret;
			var objHandle = new ctypes.unsigned_long;
			var objCount = new ctypes.unsigned_long;

			// set up to find all objects
			ret = C_FindObjectsInit(sessionHandle, null, 0);
			console.log("C_FindObjectsInit ret 0x" + ret.toString(16));




			// begin search
			if (ret == CKR_OK) {
				var i = 0;
				while (++i <= MAX_OBJECT_SEARCH) {
					ret = C_FindObjects(sessionHandle, objHandle.address(), 1, objCount.address())
					if (ret != CKR_OK) {
						console.log(" *** ERROR - C_FindObjects ret 0x" + ret.toString(16));
						break;
					}
					if (objCount.value == 0) {
						console.log(" * Find finished, no more objects found. ");
						break;
					}
					console.log("objHandle: " + objHandle);
					this.printObjectInfo(sessionHandle, objHandle);
				}
				ret = C_FindObjectsFinal(sessionHandle);
				console.log("C_FindObjectsFinal ret 0x" + ret.toString(16));
				
				return ret;
			}
		};
    }
	catch ( err ) {
		//self.port.emit("errorMessage", err + "token.js");
		return err.toString();
	}
};//endToken

function isArray(myArray) {
	return Object.prototype.toString.call(myArray) === "[object Array]";
}

function TokenExport() {
	  return new Token();
	}

exports.Token = TokenExport;

/********************************************************************************/
