/*! asn1pkcs10-1.0.0.js (c) 2014 Daniel Schoberle | kjur.github.com/jsrsasign/license
 */
/*
 * asn1pkcs10.js - ASN.1 DER encoder classes for PKCS#10 Certificate Signing Request (CSR)
 *
 * Copyright (c) 2014 Daniel Schoberle (daniel.schoberle@gmail.com)
 *
 * This software is licensed under the terms of the MIT License.
 * http://kjur.github.com/jsrsasign/license
 *
 * The above copyright and license notice shall be 
 * included in all copies or substantial portions of the Software.
 */

/**
 * @fileOverview
 * @name asn1pkcs10-1.0.js
 * @author Daniel Schoberle daniel.schoberle@gmail.com
 * @version 1.0.0 (2014-May-30)
 * @since jsrsasign 2.1
 * @license <a href="http://kjur.github.io/jsrsasign/license/">MIT License</a>
 */

/** 
 * kjur's class library name space
 * // already documented in asn1-1.0.js
 * @name KJUR
 * @namespace kjur's class library name space
 */
if (typeof KJUR == "undefined" || !KJUR) KJUR = {};

/**
 * kjur's ASN.1 class library name space
 * // already documented in asn1-1.0.js
 * @name KJUR.asn1
 * @namespace
 */
if (typeof KJUR.asn1 == "undefined" || !KJUR.asn1) KJUR.asn1 = {};

/**
 * kjur's ASN.1 class for PKCS#10 Certificate Signing Request (CSR) library name space
 * <p>
 * <h4>FEATURES</h4>
 * <ul>
 * <li>easily issue any kind of certificate</li>
 * <li>APIs are very similar to BouncyCastle library ASN.1 classes. So easy to learn.</li>
 * </ul>
 * </p>
 * <h4>PROVIDED CLASSES</h4>
 * <ul>
 * <li>{@link KJUR.asn1.pkcs10.CertificationRequest}</li>
 * <li>{@link KJUR.asn1.pkcs10.CertificationRequestInfo}</li>
 * <li>{@link KJUR.asn1.pkcs10.X500Name}</li>
 * <li>{@link KJUR.asn1.pkcs10.RDN}</li>
 * <li>{@link KJUR.asn1.pkcs10.AttributeTypeAndValue}</li>
 * <li>{@link KJUR.asn1.pkcs10.SubjectPublicKeyInfo}</li>
 * <li>{@link KJUR.asn1.pkcs10.AlgorithmIdentifier}</li>
 * <li>{@link KJUR.asn1.pkcs10.GeneralName}</li>
 * <li>{@link KJUR.asn1.pkcs10.GeneralNames}</li>
 * <li>{@link KJUR.asn1.pkcs10.OID}</li>

 * </ul>
 * NOTE: Please ignore method summary and document of this namespace. This caused by a bug of jsdoc2.
 * @name KJUR.asn1.pkcs10
 * @namespace
 */
if (typeof KJUR.asn1.pkcs10 == "undefined" || !KJUR.asn1.pkcs10) KJUR.asn1.pkcs10 = {};

// === BEGIN Certificate ===================================================

/**
 * X.509 Certificate class to sign and generate hex encoded certificate
 * @name KJUR.asn1.pkcs10.CertificationRequest
 * @class X.509 Certificate class to sign and generate hex encoded certificate
 * @param {Array} params associative array of parameters (ex. {'tbscertobj': obj, 'prvkeyobj': key})
 * @extends KJUR.asn1.ASN1Object
 * @description
 * <br/>
 * As for argument 'params' for constructor, you can specify one of
 * following properties:
 * <ul>
 * <li>tbscertobj - specify {@link KJUR.asn1.pkcs10.CertificationRequestInfo} object</li>
 * <li>prvkeyobj - specify {@link RSAKey}, {@link KJUR.crypto.ECDSA} or {@link KJUR.crypto.DSA} object for CA private key to sign the certificate</li>
 * <li>(DEPRECATED)rsaprvkey - specify {@link RSAKey} object CA private key</li>
 * <li>(DEPRECATED)rsaprvpem - specify PEM string of RSA CA private key</li>
 * </ul>
 * NOTE1: 'params' can be omitted.<br/>
 * NOTE2: DSA/ECDSA is also supported for CA signging key from asn1x509 1.0.6.
 * @example
 * var caKey = KEYUTIL.getKey(caKeyPEM); // CA's private key
 * var cert = new KJUR.asn1pkcs10.CertificationRequest({'tbscertobj': tbs, 'prvkeyobj': caKey});
 * cert.sign(); // issue certificate by CA's private key
 * var certPEM = cert.getPEMString();
 *
 * // CertificationRequest  ::=  SEQUENCE  {
 * //     certificationRequestInfo	CertificationRequestInfo,
 * //     signatureAlgorithm   		AlgorithmIdentifier,
 * //     signature            		BIT STRING  }        
 */
KJUR.asn1.pkcs10.CertificationRequest = function(params) {
    KJUR.asn1.pkcs10.CertificationRequest.superclass.constructor.call(this);
    var asn1TBSCert = null;
    var asn1SignatureAlg = null;
    var asn1Sig = null;
    var hexSig = null;
    var prvKey = null;
    var rsaPrvKey = null; // DEPRECATED

    
    /**
     * set signature algorithm field by parameter
     * @name setSignatureAlgByParam
     * @memberOf KJUR.asn1.pkcs10.CertificationRequest
     * @function
     * @param {Array} algIdParam AlgorithmIdentifier parameter
     * @description
     * @example
     * cert.setSignatureAlgByParam({'name': 'SHA1withRSA'});
     */
    this.setSignatureAlgByParam = function(algIdParam) {
        this.asn1SignatureAlg = new KJUR.asn1.pkcs10.AlgorithmIdentifier(algIdParam);
    };

    /**
     * set PKCS#5 encrypted RSA PEM private key as CA key
     * @name setRsaPrvKeyByPEMandPass
     * @memberOf KJUR.asn1.pkcs10.CertificationRequest
     * @function
     * @param {String} rsaPEM string of PKCS#5 encrypted RSA PEM private key
     * @param {String} passPEM passcode string to decrypt private key
     * @since 1.0.1
     * @description
     * <br/>
     * <h4>EXAMPLES</h4>
     * @example
     * var cert = new KJUR.asn1.pkcs10.CertificationRequest({'tbscertobj': tbs});
     * cert.setRsaPrvKeyByPEMandPass("-----BEGIN RSA PRIVATE..(snip)", "password");
     */
    this.setRsaPrvKeyByPEMandPass = function(rsaPEM, passPEM) {
        var caKeyHex = PKCS5PKEY.getDecryptedKeyHex(rsaPEM, passPEM);
        var caKey = new RSAKey();
        caKey.readPrivateKeyFromASN1HexString(caKeyHex);  
        this.prvKey = caKey;
    };

    /**
     * sign CertificationRequestInfo and set signature value internally
     * @name sign
     * @memberOf KJUR.asn1.pkcs10.CertificationRequest
     * @function
     * @description
     * @example
     * var cert = new KJUR.asn1.pkcs10.CertificationRequest({'tbscertobj': tbs, 'rsaprvkey': prvKey});
     * cert.sign();
     */
    this.sign = function() {
    	this.asn1SignatureAlg = new KJUR.asn1.pkcs10.AlgorithmIdentifier({'name': 'SHA1withRSA'});

        sig = new KJUR.crypto.Signature({'alg': 'SHA1withRSA'});
        sig.init(this.prvKey);
        sig.updateHex(this.asn1TBSCert.getEncodedHex());
        this.hexSig = sig.sign();

        this.asn1Sig = new KJUR.asn1.DERBitString({'hex': '00' + this.hexSig});
        
        var seq = new KJUR.asn1.DERSequence({'array': [this.asn1TBSCert,
                                                       this.asn1SignatureAlg,
                                                       this.asn1Sig]});
        this.hTLV = seq.getEncodedHex();
        this.isModified = false;
    };

    /**
     * set signature value internally by hex string
     * @name setSignatureHex
     * @memberOf KJUR.asn1.pkcs10.CertificationRequest
     * @function
     * @since asn1pkcs10 1.0.8
     * @description
     * @example
     * var cert = new KJUR.asn1.pkcs10.CertificationRequest({'tbscertobj': tbs});
     * cert.setSignatureHex('01020304');
     */
    this.setSignatureHex = function(sigHex) {
        this.asn1SignatureAlg = new KJUR.asn1.pkcs10.AlgorithmIdentifier({'name': 'SHA1withRSA'});
        this.hexSig = sigHex;
        this.asn1Sig = new KJUR.asn1.DERBitString({'hex': '00' + this.hexSig});

        var seq = new KJUR.asn1.DERSequence({'array': [this.asn1TBSCert,
                                                       this.asn1SignatureAlg,
                                                       this.asn1Sig]});
        this.hTLV = seq.getEncodedHex();
        this.isModified = false;
    };

    this.getEncodedHex = function() {
        if (this.isModified == false && this.hTLV != null) return this.hTLV;
        throw "not signed yet";
    };

    /**
     * get PEM formatted certificate string after signed
     * @name getPEMString
     * @memberOf KJUR.asn1.pkcs10.CertificationRequest
     * @function
     * @return PEM formatted string of certificate
     * @description
     * @example
     * var cert = new KJUR.asn1.pkcs10.CertificationRequest({'tbscertobj': tbs, 'rsaprvkey': prvKey});
     * cert.sign();
     * var sPEM =  cert.getPEMString();
     */
    this.getPEMString = function() {
        var hCert = this.getEncodedHex();
        var wCert = CryptoJS.enc.Hex.parse(hCert);
        var b64Cert = CryptoJS.enc.Base64.stringify(wCert);
        var pemBody = b64Cert.replace(/(.{64})/g, "$1\r\n");
        return "-----BEGIN CERTIFICATE REQUEST-----\r\n" + pemBody + "\r\n-----END CERTIFICATE REQUEST-----\r\n";
    };

    if (typeof params != "undefined") {
        if (typeof params['tbscertobj'] != "undefined") {
            this.asn1TBSCert = params['tbscertobj'];
        }
        if (typeof params['prvkeyobj'] != "undefined") {
            this.prvKey = params['prvkeyobj'];
        } else if (typeof params['rsaprvkey'] != "undefined") {
            this.prvKey = params['rsaprvkey'];
        } else if ((typeof params['rsaprvpem'] != "undefined") &&
                   (typeof params['rsaprvpas'] != "undefined")) {
            this.setRsaPrvKeyByPEMandPass(params['rsaprvpem'], params['rsaprvpas']);
        }
    }
};
YAHOO.lang.extend(KJUR.asn1.pkcs10.CertificationRequest, KJUR.asn1.ASN1Object);

/**
 * ASN.1 CertificationRequestInfo structure class
 * @name KJUR.asn1.pkcs10.CertificationRequestInfo
 * @class ASN.1 CertificationRequestInfo structure class
 * @param {Array} params associative array of parameters (ex. {})
 * @extends KJUR.asn1.ASN1Object
 * @description
 * <br/>
 * <h4>EXAMPLE</h4>
 * @example
 *  var o = new KJUR.asn1.pkcs10.CertificationRequestInfo();
 *  o.setSignatureAlgByParam({'name': 'SHA1withRSA'});
 *  o.setSubjectByParam({'str': '/C=US/CN=b'});
 *  o.setSubjectPublicKeyByParam({'rsakey': rsaKey});
 */
KJUR.asn1.pkcs10.CertificationRequestInfo = function(params) {
    KJUR.asn1.pkcs10.CertificationRequestInfo.superclass.constructor.call(this);

    this._initialize = function() {
        this.asn1Array = new Array();

//        this.asn1Version = 
//            new KJUR.asn1.DERTaggedObject({'obj': new KJUR.asn1.DERInteger({'int': 0})});
        this.asn1Version = 
            new KJUR.asn1.DERInteger({'int': 0});
//        this.asn1SerialNumber = null;
//        this.asn1SignatureAlg = null;
//        this.asn1Issuer = null;
//        this.asn1NotBefore = null;
//        this.asn1NotAfter = null;
        this.asn1Subject = null;
        this.asn1SubjPKey = null;
//        this.extensionsArray = new Array();
    };

    /**
     * set subject name field by parameter
     * @name setSubjectByParam
     * @memberOf KJUR.asn1.pkcs10.CertificationRequestInfo
     * @function
     * @param {Array} x500NameParam X500Name parameter
     * @description
     * @example
     * tbsc.setSubjectParam({'str': '/C=US/CN=b'});
     * @see KJUR.asn1.pkcs10.X500Name
     */
    this.setSubjectByParam = function(x500NameParam) {
        this.asn1Subject = new KJUR.asn1.pkcs10.X500Name(x500NameParam);
    };

    /**
     * (DEPRECATED) set subject public key info field by RSA key parameter
     * @name setSubjectPublicKeyByParam
     * @memberOf KJUR.asn1.pkcs10.CertificationRequestInfo
     * @function
     * @param {Array} subjPKeyParam SubjectPublicKeyInfo parameter of RSA
     * @deprecated
     * @description
     * @example
     * tbsc.setSubjectPublicKeyByParam({'rsakey': pubKey});
     * @see KJUR.asn1.pkcs10.SubjectPublicKeyInfo
     */
    this.setSubjectPublicKeyByParam = function(subjPKeyParam) {
        this.asn1SubjPKey = new KJUR.asn1.pkcs10.SubjectPublicKeyInfo(subjPKeyParam);
    };

    /**
     * set subject public key info by RSA/ECDSA/DSA key parameter
     * @name setSubjectPublicKeyByGetKey
     * @memberOf KJUR.asn1.pkcs10.CertificationRequestInfo
     * @function
     * @param {Object} keyParam public key parameter which passed to {@link KEYUTIL.getKey} argument
     * @description
     * @example
     * tbsc.setSubjectPublicKeyByGetKeyParam(certPEMString); // or 
     * tbsc.setSubjectPublicKeyByGetKeyParam(pkcs8PublicKeyPEMString); // or 
     * tbsc.setSubjectPublicKeyByGetKeyParam(kjurCryptoECDSAKeyObject); // et.al.
     * @see KJUR.asn1.pkcs10.SubjectPublicKeyInfo
     * @see KEYUTIL.getKey
     * @since asn1pkcs10 1.0.6
     */
    this.setSubjectPublicKeyByGetKey = function(keyParam) {
        var keyObj = KEYUTIL.getKey(keyParam);
        this.asn1SubjPKey = new KJUR.asn1.pkcs10.SubjectPublicKeyInfo(keyObj);
    };

    this.getEncodedHex = function() {
        this.asn1Array = new Array();

        this.asn1Array.push(this.asn1Version);
        this.asn1Array.push(this.asn1Subject);
        this.asn1Array.push(this.asn1SubjPKey);
//        this.asn1Array.push(this.extensionsArray);

        var o = new KJUR.asn1.DERSequence({"array": this.asn1Array});
        this.hTLV = o.getEncodedHex();
        this.isModified = false;
        return this.hTLV;
    };

    this._initialize();
};
YAHOO.lang.extend(KJUR.asn1.pkcs10.CertificationRequestInfo, KJUR.asn1.ASN1Object);

// === END   CertificationRequestInfo ===================================================


// === BEGIN X500Name Related =================================================
/**
 * X500Name ASN.1 structure class
 * @name KJUR.asn1.pkcs10.X500Name
 * @class X500Name ASN.1 structure class
 * @param {Array} params associative array of parameters (ex. {'str': '/C=US/O=a'})
 * @extends KJUR.asn1.ASN1Object
 * @description
 * @example
 */
KJUR.asn1.pkcs10.X500Name = function(params) {
    KJUR.asn1.pkcs10.X500Name.superclass.constructor.call(this);
    this.asn1Array = new Array();

    this.setByString = function(dnStr) {
        var a = dnStr.split('/');
        a.shift();
        for (var i = 0; i < a.length; i++) {
            this.asn1Array.push(new KJUR.asn1.pkcs10.RDN({'str':a[i]}));
        }
    };

    this.getEncodedHex = function() {
        if (typeof this.hTLV == "string") return this.hTLV;
        var o = new KJUR.asn1.DERSequence({"array": this.asn1Array});
        this.hTLV = o.getEncodedHex();
        return this.hTLV;
    };

    if (typeof params != "undefined") {
        if (typeof params['str'] != "undefined") {
            this.setByString(params['str']);
        }
        if (typeof params.certissuer != "undefined") {
            var x = new X509();
            x.hex = X509.pemToHex(params.certissuer);
            this.hTLV = x.getIssuerHex();
        }
        if (typeof params.certsubject != "undefined") {
            var x = new X509();
            x.hex = X509.pemToHex(params.certsubject);
            this.hTLV = x.getSubjectHex();
        }
    }
};
YAHOO.lang.extend(KJUR.asn1.pkcs10.X500Name, KJUR.asn1.ASN1Object);

/**
 * RDN (Relative Distinguish Name) ASN.1 structure class
 * @name KJUR.asn1.pkcs10.RDN
 * @class RDN (Relative Distinguish Name) ASN.1 structure class
 * @param {Array} params associative array of parameters (ex. {'str': 'C=US'})
 * @extends KJUR.asn1.ASN1Object
 * @description
 * @example
 */
KJUR.asn1.pkcs10.RDN = function(params) {
    KJUR.asn1.pkcs10.RDN.superclass.constructor.call(this);
    this.asn1Array = new Array();

    this.addByString = function(rdnStr) {
        this.asn1Array.push(new KJUR.asn1.pkcs10.AttributeTypeAndValue({'str':rdnStr}));
    };

    this.getEncodedHex = function() {
        var o = new KJUR.asn1.DERSet({"array": this.asn1Array});
        this.TLV = o.getEncodedHex();
        return this.TLV;
    };

    if (typeof params != "undefined") {
        if (typeof params['str'] != "undefined") {
            this.addByString(params['str']);
        }
    }
};
YAHOO.lang.extend(KJUR.asn1.pkcs10.RDN, KJUR.asn1.ASN1Object);

/**
 * AttributeTypeAndValue ASN.1 structure class
 * @name KJUR.asn1.pkcs10.AttributeTypeAndValue
 * @class AttributeTypeAndValue ASN.1 structure class
 * @param {Array} params associative array of parameters (ex. {'str': 'C=US'})
 * @extends KJUR.asn1.ASN1Object
 * @description
 * @example
 */
KJUR.asn1.pkcs10.AttributeTypeAndValue = function(params) {
    KJUR.asn1.pkcs10.AttributeTypeAndValue.superclass.constructor.call(this);
    var typeObj = null;
    var valueObj = null;
    var defaultDSType = "utf8";

    this.setByString = function(attrTypeAndValueStr) {
        if (attrTypeAndValueStr.match(/^([^=]+)=(.+)$/)) {
            this.setByAttrTypeAndValueStr(RegExp.$1, RegExp.$2);
        } else {
            throw "malformed attrTypeAndValueStr: " + attrTypeAndValueStr;
        }
    };

    this.setByAttrTypeAndValueStr = function(shortAttrType, valueStr) {
        this.typeObj = KJUR.asn1.pkcs10.OID.atype2obj(shortAttrType);
        var dsType = defaultDSType;
        if (shortAttrType == "C") dsType = "prn";
        this.valueObj = this.getValueObj(dsType, valueStr);
    };

    this.getValueObj = function(dsType, valueStr) {
        if (dsType == "utf8")   return new KJUR.asn1.DERUTF8String({"str": valueStr});
        if (dsType == "prn")    return new KJUR.asn1.DERPrintableString({"str": valueStr});
        if (dsType == "tel")    return new KJUR.asn1.DERTeletexString({"str": valueStr});
        if (dsType == "ia5")    return new KJUR.asn1.DERIA5String({"str": valueStr});
        throw "unsupported directory string type: type=" + dsType + " value=" + valueStr;
    };

    this.getEncodedHex = function() {
        var o = new KJUR.asn1.DERSequence({"array": [this.typeObj, this.valueObj]});
        this.TLV = o.getEncodedHex();
        return this.TLV;
    };

    if (typeof params != "undefined") {
        if (typeof params['str'] != "undefined") {
            this.setByString(params['str']);
        }
    }
};
YAHOO.lang.extend(KJUR.asn1.pkcs10.AttributeTypeAndValue, KJUR.asn1.ASN1Object);

// === END   X500Name Related =================================================

// === BEGIN Other ASN1 structure class  ======================================

/**
 * SubjectPublicKeyInfo ASN.1 structure class
 * @name KJUR.asn1.pkcs10.SubjectPublicKeyInfo
 * @class SubjectPublicKeyInfo ASN.1 structure class
 * @param {Object} params parameter for subject public key
 * @extends KJUR.asn1.ASN1Object
 * @description
 * <br/>
 * As for argument 'params' for constructor, you can specify one of
 * following properties:
 * <ul>
 * <li>{@link RSAKey} object</li>
 * <li>{@link KJUR.crypto.ECDSA} object</li>
 * <li>{@link KJUR.crypto.DSA} object</li>
 * <li>(DEPRECATED)rsakey - specify {@link RSAKey} object of subject public key</li>
 * <li>(DEPRECATED)rsapem - specify a string of PEM public key of RSA key</li>
 * </ul>
 * NOTE1: 'params' can be omitted.<br/>
 * NOTE2: DSA/ECDSA key object is also supported since asn1x509 1.0.6.<br/>
 * <h4>EXAMPLE</h4>
 * @example
 * var spki = new KJUR.asn1.pkcs10.SubjectPublicKeyInfo(RSAKey_object);
 * var spki = new KJUR.asn1.pkcs10.SubjectPublicKeyInfo(KJURcryptoECDSA_object);
 * var spki = new KJUR.asn1.pkcs10.SubjectPublicKeyInfo(KJURcryptoDSA_object);
 */
KJUR.asn1.pkcs10.SubjectPublicKeyInfo = function(params) {
    KJUR.asn1.pkcs10.SubjectPublicKeyInfo.superclass.constructor.call(this);
    var asn1AlgId = null;
    var asn1SubjPKey = null;
    var rsaKey = null;

    /**
     * (DEPRECATED) set RSAKey object as subject public key
     * @name setRSAKey
     * @memberOf KJUR.asn1.pkcs10.SubjectPublicKeyInfo
     * @function
     * @param {RSAKey} rsaKey {@link RSAKey} object for RSA public key
     * @description
     * @deprecated
     * @example
     * spki.setRSAKey(rsaKey);
     */
    this.setRSAKey = function(rsaKey) {
        if (! RSAKey.prototype.isPrototypeOf(rsaKey))
            throw "argument is not RSAKey instance";
        this.rsaKey = rsaKey;
        var asn1RsaN = new KJUR.asn1.DERInteger({'bigint': rsaKey.n});
        var asn1RsaE = new KJUR.asn1.DERInteger({'int': rsaKey.e});
        var asn1RsaPub = new KJUR.asn1.DERSequence({'array': [asn1RsaN, asn1RsaE]});
        var rsaKeyHex = asn1RsaPub.getEncodedHex();
        this.asn1AlgId = new KJUR.asn1.pkcs10.AlgorithmIdentifier({'name':'rsaEncryption'});
        this.asn1SubjPKey = new KJUR.asn1.DERBitString({'hex':'00'+rsaKeyHex});
    };

    /**
     * (DEPRECATED) set a PEM formatted RSA public key string as RSA public key
     * @name setRSAPEM
     * @memberOf KJUR.asn1.pkcs10.SubjectPublicKeyInfo
     * @function
     * @param {String} rsaPubPEM PEM formatted RSA public key string
     * @deprecated
     * @description
     * @example
     * spki.setRSAPEM(rsaPubPEM);
     */
    this.setRSAPEM = function(rsaPubPEM) {
        if (rsaPubPEM.match(/-----BEGIN PUBLIC KEY-----/)) {
            var s = rsaPubPEM;
            s = s.replace(/^-----[^-]+-----/, '');
            s = s.replace(/-----[^-]+-----\s*$/, '');
            var rsaB64 = s.replace(/\s+/g, '');
            var rsaWA = CryptoJS.enc.Base64.parse(rsaB64);
            var rsaP8Hex = CryptoJS.enc.Hex.stringify(rsaWA);
            var a = _rsapem_getHexValueArrayOfChildrenFromHex(rsaP8Hex);
            var hBitStrVal = a[1];
            var rsaHex = hBitStrVal.substr(2);
            var a3 = _rsapem_getHexValueArrayOfChildrenFromHex(rsaHex);
            var rsaKey = new RSAKey();
            rsaKey.setPublic(a3[0], a3[1]);
            this.setRSAKey(rsaKey);
        } else {
            throw "key not supported";
        }
    };

    /*
     * @since asn1x509 1.0.7
     */
    this.getASN1Object = function() {
        if (this.asn1AlgId == null || this.asn1SubjPKey == null)
            throw "algId and/or subjPubKey not set";
        var o = new KJUR.asn1.DERSequence({'array':
                                           [this.asn1AlgId, this.asn1SubjPKey]});
        return o;
    };

    this.getEncodedHex = function() {
        var o = this.getASN1Object();
        this.hTLV = o.getEncodedHex();
        return this.hTLV;
    };

    this._setRSAKey = function(key) {
        var asn1RsaPub = KJUR.asn1.ASN1Util.newObject({
            'seq': [{'int': {'bigint': key.n}}, {'int': {'int': key.e}}]
        });
        var rsaKeyHex = asn1RsaPub.getEncodedHex();
        this.asn1AlgId = new KJUR.asn1.pkcs10.AlgorithmIdentifier({'name':'rsaEncryption'});
        this.asn1SubjPKey = new KJUR.asn1.DERBitString({'hex':'00'+rsaKeyHex});
    };

    this._setEC = function(key) {
        var asn1Params = new KJUR.asn1.DERObjectIdentifier({'name': key.curveName});
        this.asn1AlgId = 
            new KJUR.asn1.pkcs10.AlgorithmIdentifier({'name': 'ecPublicKey',
                                                    'asn1params': asn1Params});
        this.asn1SubjPKey = new KJUR.asn1.DERBitString({'hex': '00' + key.pubKeyHex});
    };

    this._setDSA = function(key) {
        var asn1Params = new KJUR.asn1.ASN1Util.newObject({
            'seq': [{'int': {'bigint': key.p}},
                    {'int': {'bigint': key.q}},
                    {'int': {'bigint': key.g}}]
        });
        this.asn1AlgId = 
            new KJUR.asn1.pkcs10.AlgorithmIdentifier({'name': 'dsa',
                                                    'asn1params': asn1Params});
        var pubInt = new KJUR.asn1.DERInteger({'bigint': key.y});
        this.asn1SubjPKey = new KJUR.asn1.DERBitString({'hex': '00' + pubInt.getEncodedHex()});
    };

    if (typeof params != "undefined") {
        if (typeof RSAKey != 'undefined' && params instanceof RSAKey) {
            this._setRSAKey(params);
        } else if (typeof KJUR.crypto.ECDSA != 'undefined' &&
                   params instanceof KJUR.crypto.ECDSA) {
            this._setEC(params);
        } else if (typeof KJUR.crypto.DSA != 'undefined' &&
                   params instanceof KJUR.crypto.DSA) {
            this._setDSA(params);
        } else if (typeof params['rsakey'] != "undefined") {
            this.setRSAKey(params['rsakey']);
        } else if (typeof params['rsapem'] != "undefined") {
            this.setRSAPEM(params['rsapem']);
        }
    }
};
YAHOO.lang.extend(KJUR.asn1.pkcs10.SubjectPublicKeyInfo, KJUR.asn1.ASN1Object);

/**
 * AlgorithmIdentifier ASN.1 structure class
 * @name KJUR.asn1.pkcs10.AlgorithmIdentifier
 * @class AlgorithmIdentifier ASN.1 structure class
 * @param {Array} params associative array of parameters (ex. {'name': 'SHA1withRSA'})
 * @extends KJUR.asn1.ASN1Object
 * @description
 * @example
 */
KJUR.asn1.pkcs10.AlgorithmIdentifier = function(params) {
    KJUR.asn1.pkcs10.AlgorithmIdentifier.superclass.constructor.call(this);
    var nameAlg = null;
    var asn1Alg = null;
    var asn1Params = null;
    var paramEmpty = false;

    this.getEncodedHex = function() {
        if (this.nameAlg == null && this.asn1Alg == null) {
            throw "algorithm not specified";
        }
        if (this.nameAlg != null && this.asn1Alg == null) {
            this.asn1Alg = KJUR.asn1.pkcs10.OID.name2obj(this.nameAlg);
        }
        var a = [this.asn1Alg];
        if (! this.paramEmpty) a.push(this.asn1Params);
        var o = new KJUR.asn1.DERSequence({'array': a});
        this.hTLV = o.getEncodedHex();
        return this.hTLV;
    };

    if (typeof params != "undefined") {
        if (typeof params['name'] != "undefined") {
            this.nameAlg = params['name'];
        }
        if (typeof params['asn1params'] != "undefined") {
            this.asn1Params = params['asn1params'];
        }
        if (typeof params['paramempty'] != "undefined") {
            this.paramEmpty = params['paramempty'];
        }
    }
    if (this.asn1Params == null) {
        this.asn1Params = new KJUR.asn1.DERNull();
    }
};
YAHOO.lang.extend(KJUR.asn1.pkcs10.AlgorithmIdentifier, KJUR.asn1.ASN1Object);

/**
 * GeneralName ASN.1 structure class
 * @name KJUR.asn1.pkcs10.GeneralName
 * @class GeneralName ASN.1 structure class
 * @description
 * <br/>
 * As for argument 'params' for constructor, you can specify one of
 * following properties:
 * <ul>
 * <li>rfc822 - rfc822Name[1] (ex. user1@foo.com)</li>
 * <li>dns - dNSName[2] (ex. foo.com)</li>
 * <li>uri - uniformResourceIdentifier[6] (ex. http://foo.com/)</li>
 * </ul>
 * NOTE: Currently this only supports 'uniformResourceIdentifier'.
 * <h4>EXAMPLE AND ASN.1 SYNTAX</h4>
 * @example
 * var gn = new KJUR.asn1.pkcs10.GeneralName({'uri': 'http://aaa.com/'});
 *
 * GeneralName ::= CHOICE {
 *         otherName                       [0]     OtherName,
 *         rfc822Name                      [1]     IA5String,
 *         dNSName                         [2]     IA5String,
 *         x400Address                     [3]     ORAddress,
 *         directoryName                   [4]     Name,
 *         ediPartyName                    [5]     EDIPartyName,
 *         uniformResourceIdentifier       [6]     IA5String,
 *         iPAddress                       [7]     OCTET STRING,
 *         registeredID                    [8]     OBJECT IDENTIFIER } 
 */
KJUR.asn1.pkcs10.GeneralName = function(params) {
    KJUR.asn1.pkcs10.GeneralName.superclass.constructor.call(this);
    var asn1Obj = null;
    var type = null;
    var pTag = {'rfc822': '81', 'dns': '82', 'uri': '86'};

    this.setByParam = function(params) {
        var str = null;
        var v = null;

        if (typeof params['rfc822'] != "undefined") {
            this.type = 'rfc822';
            v = new KJUR.asn1.DERIA5String({'str': params[this.type]});
        }
        if (typeof params['dns'] != "undefined") {
            this.type = 'dns';
            v = new KJUR.asn1.DERIA5String({'str': params[this.type]});
        }
        if (typeof params['uri'] != "undefined") {
            this.type = 'uri';
            v = new KJUR.asn1.DERIA5String({'str': params[this.type]});
        }

        if (this.type == null)
            throw "unsupported type in params=" + params;
        this.asn1Obj = new KJUR.asn1.DERTaggedObject({'explicit': false,
                                                      'tag': pTag[this.type],
                                                      'obj': v});
    };

    this.getEncodedHex = function() {
        return this.asn1Obj.getEncodedHex();
    }

    if (typeof params != "undefined") {
        this.setByParam(params);
    }

};
YAHOO.lang.extend(KJUR.asn1.pkcs10.GeneralName, KJUR.asn1.ASN1Object);

/**
 * GeneralNames ASN.1 structure class
 * @name KJUR.asn1.pkcs10.GeneralNames
 * @class GeneralNames ASN.1 structure class
 * @description
 * <br/>
 * <h4>EXAMPLE AND ASN.1 SYNTAX</h4>
 * @example
 * var gns = new KJUR.asn1.pkcs10.GeneralNames([{'uri': 'http://aaa.com/'}, {'uri': 'http://bbb.com/'}]); 
 *
 * GeneralNames ::= SEQUENCE SIZE (1..MAX) OF GeneralName
 */
KJUR.asn1.pkcs10.GeneralNames = function(paramsArray) {
    KJUR.asn1.pkcs10.GeneralNames.superclass.constructor.call(this);
    var asn1Array = null;

    /**
     * set a array of {@link KJUR.asn1.pkcs10.GeneralName} parameters
     * @name setByParamArray
     * @memberOf KJUR.asn1.pkcs10.GeneralNames
     * @function
     * @param {Array} paramsArray Array of {@link KJUR.asn1.pkcs10.GeneralNames}
     * @description
     * <br/>
     * <h4>EXAMPLES</h4>
     * @example
     * var gns = new KJUR.asn1.pkcs10.GeneralNames();
     * gns.setByParamArray([{'uri': 'http://aaa.com/'}, {'uri': 'http://bbb.com/'}]);
     */
    this.setByParamArray = function(paramsArray) {
        for (var i = 0; i < paramsArray.length; i++) {
            var o = new KJUR.asn1.pkcs10.GeneralName(paramsArray[i]);
            this.asn1Array.push(o);
        }
    };

    this.getEncodedHex = function() {
        var o = new KJUR.asn1.DERSequence({'array': this.asn1Array});
        return o.getEncodedHex();
    };

    this.asn1Array = new Array();
    if (typeof paramsArray != "undefined") {
        this.setByParamArray(paramsArray);
    }
};
YAHOO.lang.extend(KJUR.asn1.pkcs10.GeneralNames, KJUR.asn1.ASN1Object);


/**
 * static object for OID
 * @name KJUR.asn1.pkcs10.OID
 * @class static object for OID
 * @property {Assoc Array} atype2oidList for short attribyte type name and oid (i.e. 'C' and '2.5.4.6')
 * @property {Assoc Array} name2oidList for oid name and oid (i.e. 'keyUsage' and '2.5.29.15')
 * @property {Assoc Array} objCache for caching name and DERObjectIdentifier object 
 * @description
 * <dl>
 * <dt><b>atype2oidList</b>
 * <dd>currently supports 'C', 'O', 'OU', 'ST', 'L' and 'CN' only.
 * <dt><b>name2oidList</b>
 * <dd>currently supports 'SHA1withRSA', 'rsaEncryption' and some extension OIDs
 * </dl>
 * @example
 */
KJUR.asn1.pkcs10.OID = new function(params) {
    this.atype2oidList = {
        'C':    '2.5.4.6',
        'O':    '2.5.4.10',
        'OU':   '2.5.4.11',
        'ST':   '2.5.4.8',
        'L':    '2.5.4.7',
        'CN':   '2.5.4.3',
        'DN':   '2.5.4.49',
        'DC':   '0.9.2342.19200300.100.1.25',
    };
    this.name2oidList = {
        'sha1':                 '1.3.14.3.2.26',
        'sha256':               '2.16.840.1.101.3.4.2.1',
        'sha384':               '2.16.840.1.101.3.4.2.2',
        'sha512':               '2.16.840.1.101.3.4.2.3',
        'sha224':               '2.16.840.1.101.3.4.2.4',
        'md5':                  '1.2.840.113549.2.5',
        'md2':                  '1.3.14.7.2.2.1',
        'ripemd160':            '1.3.36.3.2.1',

        'MD2withRSA':           '1.2.840.113549.1.1.2',
        'MD4withRSA':           '1.2.840.113549.1.1.3',
        'MD5withRSA':           '1.2.840.113549.1.1.4',
        'SHA1withRSA':          '1.2.840.113549.1.1.5',
        'SHA224withRSA':        '1.2.840.113549.1.1.14',
        'SHA256withRSA':        '1.2.840.113549.1.1.11',
        'SHA384withRSA':        '1.2.840.113549.1.1.12',
        'SHA512withRSA':        '1.2.840.113549.1.1.13',

        'SHA1withECDSA':        '1.2.840.10045.4.1',
        'SHA224withECDSA':      '1.2.840.10045.4.3.1',
        'SHA256withECDSA':      '1.2.840.10045.4.3.2',
        'SHA384withECDSA':      '1.2.840.10045.4.3.3',
        'SHA512withECDSA':      '1.2.840.10045.4.3.4',

        'dsa':                  '1.2.840.10040.4.1',
        'SHA1withDSA':          '1.2.840.10040.4.3',
        'SHA224withDSA':        '2.16.840.1.101.3.4.3.1',
        'SHA256withDSA':        '2.16.840.1.101.3.4.3.2',

        'rsaEncryption':        '1.2.840.113549.1.1.1',
        'subjectKeyIdentifier': '2.5.29.14',

        'countryName':          '2.5.4.6',
        'organization':         '2.5.4.10',
        'organizationalUnit':   '2.5.4.11',
        'stateOrProvinceName':  '2.5.4.8',
        'locality':             '2.5.4.7',
        'commonName':           '2.5.4.3',

        'keyUsage':             '2.5.29.15',
        'basicConstraints':     '2.5.29.19',
        'cRLDistributionPoints':'2.5.29.31',
        'certificatePolicies':  '2.5.29.32',
        'authorityKeyIdentifier':'2.5.29.35',
        'extKeyUsage':          '2.5.29.37',

        'anyExtendedKeyUsage':  '2.5.29.37.0',
        'serverAuth':           '1.3.6.1.5.5.7.3.1',
        'clientAuth':           '1.3.6.1.5.5.7.3.2',
        'codeSigning':          '1.3.6.1.5.5.7.3.3',
        'emailProtection':      '1.3.6.1.5.5.7.3.4',
        'timeStamping':         '1.3.6.1.5.5.7.3.8',
        'ocspSigning':          '1.3.6.1.5.5.7.3.9',

        'ecPublicKey':          '1.2.840.10045.2.1',
        'secp256r1':            '1.2.840.10045.3.1.7',
        'secp256k1':            '1.3.132.0.10',
        'secp384r1':            '1.3.132.0.34',

        'pkcs5PBES2':           '1.2.840.113549.1.5.13',
        'pkcs5PBKDF2':          '1.2.840.113549.1.5.12',

        'des-EDE3-CBC':         '1.2.840.113549.3.7',

        'data':                 '1.2.840.113549.1.7.1', // CMS data
        'signed-data':          '1.2.840.113549.1.7.2', // CMS signed-data
        'enveloped-data':       '1.2.840.113549.1.7.3', // CMS enveloped-data
        'digested-data':        '1.2.840.113549.1.7.5', // CMS digested-data
        'encrypted-data':       '1.2.840.113549.1.7.6', // CMS encrypted-data
        'authenticated-data':   '1.2.840.113549.1.9.16.1.2', // CMS authenticated-data
        'tstinfo':              '1.2.840.113549.1.9.16.1.4', // RFC3161 TSTInfo
    };

    this.objCache = {};

    /**
     * get DERObjectIdentifier by registered OID name
     * @name name2obj
     * @memberOf KJUR.asn1.pkcs10.OID
     * @function
     * @param {String} name OID
     * @description
     * @example
     * var asn1ObjOID = OID.name2obj('SHA1withRSA');
     */
    this.name2obj = function(name) {
        if (typeof this.objCache[name] != "undefined")
            return this.objCache[name];
        if (typeof this.name2oidList[name] == "undefined")
            throw "Name of ObjectIdentifier not defined: " + name;
        var oid = this.name2oidList[name];
        var obj = new KJUR.asn1.DERObjectIdentifier({'oid': oid});
        this.objCache[name] = obj;
        return obj;
    };

    /**
     * get DERObjectIdentifier by registered attribyte type name such like 'C' or 'CN'
     * @name atype2obj
     * @memberOf KJUR.asn1.pkcs10.OID
     * @function
     * @param {String} atype short attribute type name such like 'C' or 'CN'
     * @description
     * @example
     * var asn1ObjOID = OID.atype2obj('CN');
     */
    this.atype2obj = function(atype) {
        if (typeof this.objCache[atype] != "undefined")
            return this.objCache[atype];
        if (typeof this.atype2oidList[atype] == "undefined")
            throw "AttributeType name undefined: " + atype;
        var oid = this.atype2oidList[atype];
        var obj = new KJUR.asn1.DERObjectIdentifier({'oid': oid});
        this.objCache[atype] = obj;
        return obj;
    };
};

/*
 * @since asn1x509 1.0.9
 */
KJUR.asn1.pkcs10.OID.oid2name = function(oid) {
    var list = KJUR.asn1.pkcs10.OID.name2oidList;
    for (var name in list) {
        if (list[name] == oid) return name;
    }
    return '';
};

/**
 * X.509 certificate and CRL utilities class
 * @name KJUR.asn1.pkcs10.X509Util
 * @class X.509 certificate and CRL utilities class
 */
KJUR.asn1.pkcs10.X509Util = new function() {
    /**
     * get PKCS#8 PEM public key string from RSAKey object
     * @name getPKCS8PubKeyPEMfromRSAKey
     * @memberOf KJUR.asn1.pkcs10.X509Util
     * @function
     * @param {RSAKey} rsaKey RSA public key of {@link RSAKey} object
     * @description
     * @example
     * var pem = KJUR.asn1.pkcs10.X509Util.getPKCS8PubKeyPEMfromRSAKey(pubKey);
     */
    this.getPKCS8PubKeyPEMfromRSAKey = function(rsaKey) {
        var pem = null;
        var hN = KJUR.asn1.ASN1Util.bigIntToMinTwosComplementsHex(rsaKey.n);
        var hE = KJUR.asn1.ASN1Util.integerToByteHex(rsaKey.e);
        var iN = new KJUR.asn1.DERInteger({hex: hN});
        var iE = new KJUR.asn1.DERInteger({hex: hE});
        var asn1PubKey = new KJUR.asn1.DERSequence({array: [iN, iE]});
        var hPubKey = asn1PubKey.getEncodedHex();
        var o1 = new KJUR.asn1.pkcs10.AlgorithmIdentifier({name: 'rsaEncryption'});
        var o2 = new KJUR.asn1.DERBitString({hex: '00' + hPubKey});
        var seq = new KJUR.asn1.DERSequence({array: [o1, o2]});
        var hP8 = seq.getEncodedHex();
        var pem = KJUR.asn1.ASN1Util.getPEMStringFromHex(hP8, "PUBLIC KEY");
        return pem;
    };
};


/*
  org.bouncycastle.asn1.x500
  AttributeTypeAndValue
  DirectoryString
  RDN
  X500Name
  X500NameBuilder

  org.bouncycastleasn1.x509
  TBSCertificate
*/
